package net.bluemind.backend.mail.replica.service.internal.hooks;

import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.backend.mail.replica.api.IDbByContainerReplicatedMailboxes;
import net.bluemind.backend.mail.replica.api.IDbReplicatedMailboxes;
import net.bluemind.backend.mail.replica.api.IMailReplicaUids;
import net.bluemind.core.container.api.ContainerQuery;
import net.bluemind.core.container.api.IContainers;
import net.bluemind.core.container.api.ISharedContainers;
import net.bluemind.core.container.hooks.IAclHook;
import net.bluemind.core.container.model.ContainerDescriptor;
import net.bluemind.core.container.model.DataLocation;
import net.bluemind.core.container.model.acl.AccessControlEntry;
import net.bluemind.core.container.model.acl.Verb;
import net.bluemind.core.container.repository.ISharedContainerStore;
import net.bluemind.core.container.service.internal.AclService;
import net.bluemind.core.rest.BmContext;
import net.bluemind.directory.api.DirEntry;
import net.bluemind.directory.api.IDirectory;
import net.bluemind.mailbox.api.IMailboxAclUids;
import net.bluemind.repository.provider.RepositoryProvider;

public class VisibleOnHierarchyAclHook implements IAclHook {
	private static final Logger logger = LoggerFactory.getLogger(VisibleOnHierarchyAclHook.class);

	@Override
	public void onAclChanged(BmContext context, ContainerDescriptor descriptor, List<AccessControlEntry> previous,
			List<AccessControlEntry> current) {
		/***
		 * if container type is mailbox_records,
		 * 
		 * If mailboxacl container is present for context,
		 * 
		 * If a new entry with verb Read or Write is present with subject not the
		 * context owner,
		 * 
		 * If Visible is not set for new subject entry,
		 * 
		 * Then add new verb Visible for new subject entry for context container
		 * mailboxacl
		 * 
		 ***/

		if (IMailReplicaUids.MAILBOX_RECORDS.equals(descriptor.type)) {
			List<AccessControlEntry> newEntries = new ArrayList<>(current);
			newEntries.removeAll(previous);

			List<AccessControlEntry> oldEntries = new ArrayList<>(previous);
			oldEntries.removeAll(current);
			List<AccessControlEntry> oldEntriesForOtherSubjects = oldEntries.stream()
					.filter(a -> !a.subject.equals(context.getSecurityContext().getOwnerPrincipal())).toList();
			List<AccessControlEntry> newEntriesForOtherSubjects = newEntries.stream()
					.filter(a -> !a.subject.equals(context.getSecurityContext().getOwnerPrincipal())).toList();

			IContainers containersService = context.su().provider().instance(IContainers.class);
			ContainerDescriptor aclContainerDesc = containersService
					.get(IMailboxAclUids.uidForMailbox(descriptor.owner));
			AclService aclService = new AclService(context, context.getSecurityContext(), aclContainerDesc);

			if (!oldEntriesForOtherSubjects.isEmpty()) {
				manageOldEntries(context, descriptor, aclService, oldEntriesForOtherSubjects);
			}

			if (!newEntriesForOtherSubjects.isEmpty()) {
				manageNewEntries(context, descriptor, aclService, newEntriesForOtherSubjects);
			}

		}

		/**
		 * Manage case when Visible is no more present, but a folder is still shared
		 */
		if (IMailboxAclUids.TYPE.equals(descriptor.type)) {
			ISharedContainers sharedApi = context.provider().instance(ISharedContainers.class, descriptor.domainUid,
					descriptor.owner);
			Set<AccessControlEntry> sharedContainersGivenRights = sharedApi
					.getSharedContainers(IMailReplicaUids.MAILBOX_RECORDS).stream().map(s -> s.givenRights)
					.flatMap(Set::stream).collect(Collectors.toSet());

			Set<String> subjectsForSharedFolders = sharedContainersGivenRights.stream().map(acl -> acl.subject)
					.collect(Collectors.toSet());

			previous.stream().map(a -> a.getSubject())
					.filter(s -> !s.equals(IMailboxAclUids.mailboxForUid(descriptor.uid))).collect(Collectors.toSet())
					.forEach(subject -> {
						// Folder is still shared for subject and Visible is no more present
						if (isNoMoreVisibleForSubject(current, previous, subject)
								&& subjectsForSharedFolders.contains(subject)) {
							AclService aclService = new AclService(context, context.getSecurityContext(), descriptor);
							aclService.add(List.of(AccessControlEntry.create(subject, Verb.Visible)));
						}
					});
		}
	}

	private boolean isNoMoreVisibleForSubject(List<AccessControlEntry> current, List<AccessControlEntry> previous,
			String subject) {
		boolean visibleCurrent = current.stream()
				.anyMatch(a -> a.getVerb().can(Verb.Visible) && a.getSubject().equals(subject));

		boolean visiblePrevious = previous.stream()
				.anyMatch(a -> a.getVerb().can(Verb.Visible) && a.getSubject().equals(subject));
		return visiblePrevious && !visibleCurrent;

	}

	private void manageOldEntries(BmContext context, ContainerDescriptor descriptor, AclService aclService,
			List<AccessControlEntry> list) {
		IContainers containerService = context.getServiceProvider().instance(IContainers.class);
		List<Long> mbxRecordsIds = containerService
				.all(ContainerQuery.ownerAndType(descriptor.owner, IMailReplicaUids.MAILBOX_RECORDS)).stream()
				.map(c -> c.internalId).toList();
		ISharedContainerStore sharedContainerStore = RepositoryProvider.instance(ISharedContainerStore.class, context,
				DataLocation.of(descriptor.datalocation));

		// Must remove Visible if only subjet is no more present for mailbox_records
		Set<String> subjects = list.stream().map(acl -> acl.subject).collect(Collectors.toSet());
		for (String subject : subjects) {
			try {
				List<Long> aclIds = sharedContainerStore.getForSubject(subject).stream()
						.map(result -> result.desc().internalId).toList();
				List<Long> intersection = aclIds.stream().filter(mbxRecordsIds::contains).toList();

				// Intersection is empty: must remove Visible right for container owner
				if (intersection.isEmpty()) {
					removeVisibleRightOnHierarchy(aclService, subject);
				}
			} catch (SQLException e) {
				logger.error(e.getMessage());
			}
		}
	}

	private void manageNewEntries(BmContext context, ContainerDescriptor descriptor, AclService aclService,
			List<AccessControlEntry> list) {
		Set<String> subjects = list.stream().map(acl -> acl.subject).collect(Collectors.toSet());

		List<AccessControlEntry> aclEntries = aclService.get();

		Map<String, Set<Verb>> indexedAcls = aclEntries.stream().collect(Collectors.groupingBy(
				AccessControlEntry::getSubject, Collectors.mapping(AccessControlEntry::getVerb, Collectors.toSet())));

		for (String subject : subjects) {
			if (!indexedAcls.computeIfAbsent(subject, s -> Collections.emptySet()).contains(Verb.Visible)) {
				aclService.add(List.of(AccessControlEntry.create(subject, Verb.Visible)));
				touchMailboxFolderItem(context, descriptor);
			}
		}
	}

	private void removeVisibleRightOnHierarchy(AclService aclService, String ownerUid) {
		List<AccessControlEntry> toKeep = aclService.get().stream()
				.filter(acl -> !(acl.verb.equals(Verb.Visible) && acl.subject.equals(ownerUid))).toList();
		aclService.store(toKeep);
	}

	private void touchMailboxFolderItem(BmContext context, ContainerDescriptor descriptor) {

		IDirectory dirApi = context.su().provider().instance(IDirectory.class, descriptor.domainUid);
		DirEntry owner = dirApi.findByEntryUid(descriptor.owner);

		String subtree = IMailReplicaUids.subtreeUid(descriptor.domainUid, owner);
		IDbReplicatedMailboxes subtreeApi = context.su().provider().instance(IDbByContainerReplicatedMailboxes.class,
				subtree);
		subtreeApi.touch(IMailReplicaUids.uniqueId(descriptor.uid));
	}

}
