/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2017
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.backend.mail.replica.service.internal;

import java.sql.SQLException;
import java.util.List;
import java.util.Optional;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.Lists;

import net.bluemind.backend.mail.api.IReadOnlyMailboxFolders;
import net.bluemind.backend.mail.api.MailboxFolder;
import net.bluemind.backend.mail.replica.api.AppendTx;
import net.bluemind.backend.mail.replica.api.IDbByContainerReplicatedMailboxes;
import net.bluemind.backend.mail.replica.api.IDbMailboxRecords;
import net.bluemind.backend.mail.replica.api.IDbReplicatedMailboxes;
import net.bluemind.backend.mail.replica.api.IMailReplicaUids;
import net.bluemind.backend.mail.replica.api.MailboxReplica;
import net.bluemind.backend.mail.replica.api.MailboxReplicaRootDescriptor;
import net.bluemind.backend.mail.replica.api.MailboxReplicaRootDescriptor.Namespace;
import net.bluemind.backend.mail.replica.api.SubtreeLocation;
import net.bluemind.backend.mail.replica.service.names.MailboxNameValidator;
import net.bluemind.backend.mail.replica.utils.SubtreeContainerItemIdsCache;
import net.bluemind.backend.mail.repository.IMailboxReplicaStore;
import net.bluemind.backend.mail.repository.IReplicasStore;
import net.bluemind.core.api.fault.ErrorCode;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.api.Ack;
import net.bluemind.core.container.api.ContainerHierarchyNode;
import net.bluemind.core.container.api.IContainers;
import net.bluemind.core.container.api.ItemValueExists;
import net.bluemind.core.container.hierarchy.hook.HierarchyIdsHints;
import net.bluemind.core.container.model.BaseContainerDescriptor;
import net.bluemind.core.container.model.Container;
import net.bluemind.core.container.model.ContainerDescriptor;
import net.bluemind.core.container.model.ContainerModifiableDescriptor;
import net.bluemind.core.container.model.Item;
import net.bluemind.core.container.model.ItemFlag;
import net.bluemind.core.container.model.ItemIdentifier;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.container.model.ItemVersion;
import net.bluemind.core.container.repository.IAclStore;
import net.bluemind.core.container.repository.IContainerStore;
import net.bluemind.core.container.service.internal.ContainerStoreService;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.directory.api.DirEntry;
import net.bluemind.directory.api.IDirectory;
import net.bluemind.indexing.incremental.TriggerIndexing;
import net.bluemind.repository.provider.RepositoryProvider;

public class DbReplicatedMailboxesService extends BaseReplicatedMailboxesService
		implements IDbReplicatedMailboxes, IDbByContainerReplicatedMailboxes, IReadOnlyMailboxFolders {

	private static final Logger logger = LoggerFactory.getLogger(DbReplicatedMailboxesService.class);

	public DbReplicatedMailboxesService(MailboxReplicaRootDescriptor root, Container cont, BmContext context,
			IMailboxReplicaStore store, ContainerStoreService<MailboxReplica> mboxReplicaStore,
			IContainerStore contStore, IAclStore aclStore) {
		super(root, cont, context, store, mboxReplicaStore, contStore, aclStore);
	}

	@Override
	public Ack create(String uid, MailboxReplica r) {
		logger.info("CREATE {} n:{} fn:{}", uid, r.name, r.fullName);
		MailboxReplica replica = nameSanitizer.sanitizeNames(r);
		if (!MailboxNameValidator.validate(replica)) {
			throw new ServerFault("MailboxName validator failed '" + r.fullName + "' is not allowed");
		}
		String recordsContainerUid = IMailReplicaUids.mboxRecords(uid);
		String domainUid = container.domainUid;

		Long expectedId = FolderInternalIdCache.expectedFolderId(container, replica.fullName);

		ItemVersion created = null;
		if (expectedId != null) {
			logger.info("Hierarchy will use expected id {}", expectedId);
			String hierUid = ContainerHierarchyNode.uidFor(recordsContainerUid, IMailReplicaUids.MAILBOX_RECORDS,
					domainUid);
			HierarchyIdsHints.putHint(hierUid, expectedId);
		}

		if (root.ns == Namespace.deleted || root.ns == Namespace.deletedShared) {
			replica.deleted = true;
		}

		// create the records container
		IContainers containerService = context.su().provider().instance(IContainers.class);
		BaseContainerDescriptor contDesc = containerService.getLightIfPresent(recordsContainerUid);
		if (contDesc == null) {
			ContainerDescriptor toCreate = ContainerDescriptor.create(recordsContainerUid, replica.name,
					container.owner, IMailReplicaUids.MAILBOX_RECORDS, domainUid, false);
			contDesc = containerService.create(recordsContainerUid, toCreate);
			logger.info("Records container created {}", recordsContainerUid);
		} else {
			logger.warn("Associated records container {} already exists", recordsContainerUid);
		}
		replica.uidValidity = contDesc.internalId;

		Long subtreeItemId = SubtreeContainerItemIdsCache.getFolderId(container.uid, replica.fullName);
		if (subtreeItemId == null) {
			created = storeService.create(uid, replica.name, replica);
		} else {
			SubtreeContainerItemIdsCache.removeFolderId(container.uid, replica.fullName);
			created = storeService.createWithId(uid, subtreeItemId, null, replica.name, replica);
		}

		ItemIdentifier iid = ItemIdentifier.of(uid, created.id, created.version, created.timestamp);

		String fn = replica.fullName;
		if (root.ns == Namespace.shared && replica.parentUid != null) {
			IReplicasStore repStore = RepositoryProvider.instance(IReplicasStore.class, context);
			Optional<SubtreeLocation> optRecordsLocation = SubtreeLocations.getById(repStore, replica.parentUid);
			if (!optRecordsLocation.isPresent()) {
				throw ServerFault.notFound("subtree loc not found for parent " + replica.parentUid);
			}
			SubtreeLocation recLoc = optRecordsLocation.get();
			fn = Location.imapPath(recLoc, context) + "/" + replica.name;
		}

		TriggerIndexing.forContainer(container);
		EmitReplicationEvents.mailboxCreated(container.uid, fn, iid);
		EmitReplicationEvents.subtreeUpdated(container.uid, container.owner, iid);
		return created.ack();
	}

	private void create(Item item, MailboxReplica r) {
		String uid = item.uid;
		MailboxReplica replica = nameSanitizer.sanitizeNames(r);
		String recordsContainerUid = IMailReplicaUids.mboxRecords(uid);
		String domainUid = container.domainUid;

		// create the records container
		IContainers containerService = context.su().provider().instance(IContainers.class);
		BaseContainerDescriptor contDesc = containerService.getLightIfPresent(recordsContainerUid);
		if (contDesc == null) {
			ContainerDescriptor toCreate = ContainerDescriptor.create(recordsContainerUid, replica.name,
					container.owner, IMailReplicaUids.MAILBOX_RECORDS, domainUid, false);
			contDesc = containerService.create(recordsContainerUid, toCreate);
			logger.info("Records container created {}", recordsContainerUid);
		} else {
			logger.warn("Associated records container {} already exists", recordsContainerUid);
		}
		replica.uidValidity = contDesc.internalId;

		storeService.create(item, replica);
	}

	@Override
	public Ack update(String uid, MailboxReplica r) {
		if (!MailboxNameValidator.validate(r)) {
			throw new ServerFault("MailboxName validator failed '" + r.fullName + "' is not allowed");
		}
		ItemValue<MailboxReplica> previous = getCompleteReplica(uid);
		if (previous != null) {
			IContainers contApi = context.su().provider().instance(IContainers.class);
			String recordsContainerUid = IMailReplicaUids.mboxRecords(uid);
			if (r.uidValidity == 0) {
				BaseContainerDescriptor contDesc = contApi.getLight(recordsContainerUid);
				r.uidValidity = contDesc.internalId;
			}

			MailboxReplica replica = nameSanitizer.sanitizeNames(r);
			replica.lastUid = previous.value.lastUid;
			replica.lastAppendDate = previous.value.lastAppendDate;
			ItemVersion upd = storeService.update(uid, replica.name, replica);

			ItemValue<MailboxReplica> toCache = ItemValue.create(previous, replica);
			toCache.displayName = replica.name;
			toCache.value.dataLocation = previous.value.dataLocation;
			MboxReplicasCache.cache(toCache);

			boolean minorChange = isMinorChange(replica, previous);
			if (!minorChange) {
				SubtreeLocation sl = SubtreeLocations.locations.getIfPresent(uid);
				if (sl != null) {
					String oldName = sl.boxName;
					sl.boxName = replica.fullName;
					logger.info("Updating cached location for {} from {} to {}", uid, oldName, sl.boxName);
				}
				ContainerModifiableDescriptor cmd = new ContainerModifiableDescriptor();
				cmd.deleted = replica.deleted;
				cmd.name = replica.name;
				contApi.update(recordsContainerUid, cmd);
			}

			EmitReplicationEvents.subtreeUpdated(container.uid, container.owner,
					ItemIdentifier.of(uid, upd.id, upd.version, upd.timestamp), minorChange);
			return upd.ack();
		} else {
			logger.warn("Nothing to update at uid {}", uid);
			return Ack.create(0, null);
		}
	}

	private void update(Item item, MailboxReplica r) {
		String uid = item.uid;
		ItemValue<MailboxReplica> previous = getCompleteReplica(uid);
		if (previous != null) {
			IContainers contApi = context.su().provider().instance(IContainers.class);
			String recordsContainerUid = IMailReplicaUids.mboxRecords(uid);
			if (r.uidValidity == 0) {
				BaseContainerDescriptor contDesc = contApi.getLight(recordsContainerUid);
				r.uidValidity = contDesc.internalId;
			}

			MailboxReplica replica = nameSanitizer.sanitizeNames(r);
			replica.lastUid = previous.value.lastUid;
			replica.lastAppendDate = previous.value.lastAppendDate;
			storeService.update(item, replica.name, replica);

			ItemValue<MailboxReplica> toCache = ItemValue.create(item, replica);
			toCache.displayName = replica.name;
			toCache.value.dataLocation = previous.value.dataLocation;
			MboxReplicasCache.cache(toCache);

			boolean minorChange = isMinorChange(replica, previous);
			if (!minorChange) {
				SubtreeLocation sl = SubtreeLocations.locations.getIfPresent(uid);
				if (sl != null) {
					String oldName = sl.boxName;
					sl.boxName = replica.fullName;
					logger.info("Updating cached location for {} from {} to {}", uid, oldName, sl.boxName);
				}
				ContainerModifiableDescriptor cmd = new ContainerModifiableDescriptor();
				cmd.deleted = replica.deleted;
				cmd.name = replica.name;
				contApi.update(recordsContainerUid, cmd);
			}
		} else {
			logger.warn("Nothing to update at uid {}", uid);
		}
	}

	@Override
	public void restore(ItemValue<MailboxReplica> item, boolean isCreate) {
		logger.info("RESTORE mail-folder {}", item.item());
		if (isCreate) {
			create(item.item(), item.value);
		} else {
			update(item.item(), item.value);
		}
	}

	@Override
	public ItemValueExists itemValueExists(String uid) {
		return storeService.exists(uid);
	}

	@Override
	public MailboxReplica get(String uid) {
		return Optional.ofNullable(getCompleteReplica(uid)).map(iv -> iv.value).orElse(null);
	}

	@Override
	public ItemVersion touch(String uid) {
		ItemVersion itemVersion = storeService.touch(uid);
		MboxReplicasCache.invalidate(uid);
		return itemVersion;
	}

	private boolean isMinorChange(MailboxReplica replica, ItemValue<MailboxReplica> previous) {
		boolean sameName = previous.value.fullName.equals(replica.fullName);
		boolean deletionStatusUnchanged = previous.flags.contains(ItemFlag.Deleted) == replica.deleted;

		return sameName && deletionStatusUnchanged;
	}

	private void delete0(String uid, boolean moveToTrash) {
		ItemValue<MailboxFolder> toDelete = getComplete(uid);
		if (toDelete == null) {
			MboxReplicasCache.invalidate(uid);
			return;
		}

		IDbMailboxRecords recordsApi = null;
		try {
			recordsApi = context.provider().instance(IDbMailboxRecords.class, uid);
			if (moveToTrash) {
				DirEntry dirEntry = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM)
						.instance(IDirectory.class, container.domainUid).findByEntryUid(container.owner);
				String subtreeContainer = IMailReplicaUids.subtreeUid(container.domainUid, dirEntry);
				Trash trash = new Trash(context, subtreeContainer, recordsApi);
				Lists.partition(recordsApi.allIds(), 1000).stream()
						.forEach(ids -> trash.deleteItems(toDelete.internalId, ids));
			}
			recordsApi.prepareContainerDelete();
			// is it root ??? should we drop the subtree ?
			deleteMailbox(context, uid, storeService, recordsApi, container);
		} catch (ServerFault sf) {
			logger.error("Unable to instanciate IDbMailboxRecords for uid={}: {}", uid, sf.getMessage());
		}

		TriggerIndexing.forContainer(container);
		EmitReplicationEvents.subtreeUpdated(container.uid, container.owner,
				ItemIdentifier.of(uid, toDelete.internalId, toDelete.version, toDelete.timestamp()));
	}

	@Override
	public void delete(String uid) {
		delete0(uid, true);
	}

	/**
	 * Same as delete(uid) but without going through the Trash folder, which we will
	 * remove anyway.
	 *
	 * This is used to speedup user removal
	 */
	@Override
	public void deleteNoTrash(String uid) {
		delete0(uid, false);
	}

	static void deleteMailbox(BmContext context, String uid, ContainerStoreService<MailboxReplica> storeService,
			IDbMailboxRecords recordsApi, Container container) {
		ItemVersion deleted = storeService.delete(uid);
		if (deleted != null) {
			String toDeleteContainerUid = IMailReplicaUids.mboxRecords(uid);
			if (recordsApi != null) {
				logger.info("Purge records in {} {}...", uid, toDeleteContainerUid);
				recordsApi.deleteAll();
			}

			context.su().provider().instance(IContainers.class).delete(toDeleteContainerUid);
			MboxReplicasCache.invalidate(uid);
		}
	}

	@Override
	public ItemValue<MailboxReplica> byReplicaName(String name) {
		return super.byReplicaName(name);
	}

	@Override
	public List<ItemValue<MailboxReplica>> allReplicas() {
		List<ItemValue<MailboxReplica>> ret = storeService.all();
		ret.forEach(iv -> iv.value.dataLocation = dataLocation);
		return ret;
	}

	@Override
	public ItemValue<MailboxReplica> getCompleteById(long id) {
		return storeService.get(id, null);
	}

	@Override
	public List<ItemValue<MailboxReplica>> multipleGetById(List<Long> ids) {
		return storeService.getMultipleById(ids);
	}

	@Override
	public AppendTx prepareAppend(long mboxReplicaId, Integer count) {
		int bump = count == null ? 1 : count.intValue();
		if (bump < 1) {
			throw new ServerFault("Append count must be >= 1 (" + bump + " asked)", ErrorCode.INVALID_PARAMETER);
		}
		ItemValue<MailboxReplica> item = storeService.get(mboxReplicaId, null);
		if (item == null) {
			throw ServerFault.notFound("Missing replicated mailbox with id " + mboxReplicaId);
		}
		try {
			AppendTx ret = replicaStore.prepareAppend(bump, mboxReplicaId);
			storeService.touch(item.uid);
			MboxReplicasCache.invalidate(item.uid);
			return ret;
		} catch (SQLException e) {
			throw ServerFault.sqlFault(e);
		}
	}

}
