/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2017
 *
 * This file is part of Blue Mind. Blue Mind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License)
 * or the CeCILL as published by CeCILL.info (version 2 of the License).
 *
 * There are special exceptions to the terms and conditions of the
 * licenses as they are applied to this program. See LICENSE.txt in
 * the directory of this program distribution.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.backend.mail.replica.service;

import java.sql.SQLException;

import net.bluemind.backend.mail.api.MailboxFolder;
import net.bluemind.backend.mail.replica.api.IDbByContainerReplicatedMailboxes;
import net.bluemind.backend.mail.replica.api.IDbMessageBodies;
import net.bluemind.backend.mail.replica.api.IDbReplicatedMailboxes;
import net.bluemind.backend.mail.replica.api.IMailReplicaUids;
import net.bluemind.backend.mail.replica.api.MailboxRecord;
import net.bluemind.backend.mail.replica.indexing.IMailIndexService;
import net.bluemind.backend.mail.replica.service.internal.RecordsItemFlagProvider;
import net.bluemind.backend.mail.repository.IMailboxRecordStore;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.BaseContainerDescriptor;
import net.bluemind.core.container.model.Container;
import net.bluemind.core.container.model.ContainerUid;
import net.bluemind.core.container.model.DataLocation;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.container.repository.IContainerRouteStore;
import net.bluemind.core.container.repository.IContainerStore;
import net.bluemind.core.container.repository.IWeightProvider;
import net.bluemind.core.container.service.internal.ContainerStoreService;
import net.bluemind.core.container.service.internal.ContainerStoreService.IWeightSeedProvider;
import net.bluemind.core.container.service.internal.ItemValueAuditLogService;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.directory.api.DirEntry;
import net.bluemind.directory.api.IDirectory;
import net.bluemind.repository.provider.RepositoryProvider;

public abstract class AbstractMailboxRecordServiceFactory<T>
		implements ServerSideServiceProvider.IServerSideServiceFactory<T> {

	private final RecordsItemFlagProvider flagsProvider;

	private final IWeightSeedProvider<MailboxRecord> recordSeedProvider;
	private final IWeightProvider toWeight;

	protected AbstractMailboxRecordServiceFactory() {
		this.flagsProvider = new RecordsItemFlagProvider();
		this.recordSeedProvider = rec -> rec.internalDate.getTime();
		this.toWeight = seed -> seed;
	}

	protected abstract T create(Container cont, BmContext context, String mailboxUniqueId,
			IMailboxRecordStore recordStore, ContainerStoreService<MailboxRecord> storeService,
			IMailIndexService indexService, IDbMessageBodies bodiesApi);

	private T getService(BmContext context, String mailboxUniqueId) {
		String uid = IMailReplicaUids.mboxRecords(mailboxUniqueId);
		IContainerRouteStore router = RepositoryProvider.instance(IContainerRouteStore.class, context);
		DataLocation loc = router.routeOf(ContainerUid.of(uid));
		try {
			IContainerStore cs = RepositoryProvider.instance(IContainerStore.class, context, loc);
			Container recordsContainer = cs.get(uid);
			if (recordsContainer == null) {
				throw ServerFault.notFound(
						"container for mailbox unique id=" + mailboxUniqueId + " (uid=" + uid + ") not found");
			}

			DirEntry owner = context.su().provider().instance(IDirectory.class, recordsContainer.domainUid)
					.findByEntryUid(recordsContainer.owner);
			if (owner == null) {
				throw ServerFault
						.notFound("owner " + recordsContainer.owner + "@" + recordsContainer.domainUid + " not found");
			}
			String subtreeContainerUid = IMailReplicaUids.subtreeUid(recordsContainer.domainUid, owner);
			Container subtreeContainer = cs.get(subtreeContainerUid);
			if (subtreeContainer == null) {
				throw ServerFault.notFound("subtree uid=" + subtreeContainerUid + " for mailbox unique id="
						+ mailboxUniqueId + " not found");
			}
			IMailboxRecordStore recordStore = RepositoryProvider.instance(IMailboxRecordStore.class, context,
					recordsContainer, subtreeContainer);

			BaseContainerDescriptor descriptor = BaseContainerDescriptor.create(recordsContainer.uid,
					recordsContainer.name, recordsContainer.owner, recordsContainer.type, recordsContainer.domainUid,
					recordsContainer.defaultContainer);
			descriptor.internalId = recordsContainer.id;

			IDbReplicatedMailboxes foldersApi = context.su().provider()
					.instance(IDbByContainerReplicatedMailboxes.class, subtreeContainerUid);
			String recordsFolderName = IMailReplicaUids.uniqueId(recordsContainer.uid);
			ItemValue<MailboxFolder> recordsFolder = foldersApi.getComplete(recordsFolderName);

			if (recordsFolder == null) {
				throw ServerFault.notFound("mailbox_replica unique_id=" + recordsFolderName + " not found");
			}

			IRecordServiceProvider serviceProvider = RecordServiceProvider.get(recordsFolder.value.fullName);
			IDbMessageBodies bodiesApi = serviceProvider.getBodiesService(context, loc.serverUid());
			ItemValueAuditLogService<MailboxRecord> logService = serviceProvider.getAuditlogService(context,
					descriptor);
			IMailIndexService mailIndexService = serviceProvider.getMailIndexService();

			ContainerStoreService<MailboxRecord> storeService = new HookMailboxRecordStoreService(context,
					recordsContainer, recordStore, flagsProvider, recordSeedProvider, toWeight, logService);

			return create(recordsContainer, context, mailboxUniqueId, recordStore, storeService, mailIndexService,
					bodiesApi);
		} catch (SQLException e) {
			throw ServerFault.sqlFault(e);
		}
	}

	@Override
	public T instance(BmContext context, String... params) throws ServerFault {
		if (params == null || params.length < 1) {
			throw new ServerFault("wrong number of instance parameters");
		}
		String mboxUniqueId = params[0];
		return getService(context, mboxUniqueId);
	}

}
