/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2018
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.backend.mail.replica.service.internal.repair;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;

import org.apache.james.mime4j.dom.Message;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.Lists;

import net.bluemind.backend.cyrus.partitions.CyrusPartition;
import net.bluemind.backend.mail.api.MessageBody;
import net.bluemind.backend.mail.api.flags.MailboxItemFlag;
import net.bluemind.backend.mail.parsing.EmlBuilder;
import net.bluemind.backend.mail.replica.api.IDbMailboxRecords;
import net.bluemind.backend.mail.replica.api.IDbMessageBodies;
import net.bluemind.backend.mail.replica.api.IMailReplicaUids;
import net.bluemind.backend.mail.replica.api.MailboxRecord;
import net.bluemind.backend.mail.replica.api.MailboxReplica;
import net.bluemind.backend.mail.replica.api.RawImapBinding;
import net.bluemind.backend.mail.replica.api.WithId;
import net.bluemind.core.api.fault.ErrorCode;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.task.service.IServerTaskMonitor;
import net.bluemind.directory.api.BaseDirEntry.Kind;
import net.bluemind.directory.api.MaintenanceOperation;
import net.bluemind.directory.service.IDirEntryRepairSupport;
import net.bluemind.directory.service.RepairTaskMonitor;
import net.bluemind.jna.utils.MemfdSupport;
import net.bluemind.jna.utils.OffHeapTemporaryFile;
import net.bluemind.mailbox.api.Mailbox;
import net.bluemind.mime4j.common.Mime4JHelper;
import net.bluemind.sds.dto.ExistRequest;
import net.bluemind.sds.dto.PutRequest;
import net.bluemind.sds.store.ISdsSyncStore;
import net.bluemind.sds.store.loader.SdsStoreLoader;
import net.bluemind.system.sysconf.helper.SysConfHelper;

public class MissingSdsBodiesRepair implements IDirEntryRepairSupport {

	private static final Logger logger = LoggerFactory.getLogger(MissingSdsBodiesRepair.class);

	public static class RepairFactory implements IDirEntryRepairSupport.Factory {
		@Override
		public IDirEntryRepairSupport create(BmContext context) {
			return new MissingSdsBodiesRepair(context);
		}
	}

	public static final MaintenanceOperation repairOp = MaintenanceOperation
			.create(IMailReplicaUids.REPAIR_MISSING_BODIES, "Replace missing bodies in SDS");

	private static class UploadBodyPlaceholders extends MailboxFoldersRepairOp {

		public UploadBodyPlaceholders(BmContext ctx) {
			super(ctx, repairOp.identifier, null, IMailReplicaUids.REPAIR_SUBTREE_OP, 1);
		}

		@Override
		protected void runOnFolders(boolean repair, RepairTaskMonitor mon, String subTree, String domainUid,
				ItemValue<Mailbox> mbox, List<ItemValue<MailboxReplica>> fullList) {
			mon.begin(fullList.size(), "Checking bodies in " + fullList.size() + " folder(s)");
			Optional<ISdsSyncStore> sdsStore = new SdsStoreLoader().forSysconf(SysConfHelper.fromSharedMap(),
					mbox.value.dataLocation);
			AtomicInteger repaired = new AtomicInteger();
			for (var f : fullList) {
				CyrusPartition part = CyrusPartition.forServerAndDomain(mbox.value.dataLocation, domainUid);
				IDbMessageBodies bodies = context.provider().instance(IDbMessageBodies.class, part.name);
				IDbMailboxRecords recApi = context.provider().instance(IDbMailboxRecords.class, f.uid);
				ServerFault.onExceptionVoid(() -> {
					List<RawImapBinding> allMsg = recApi.imapIdSet("1:*", "");
					if (!allMsg.isEmpty()) {
						IServerTaskMonitor inFolder = mon.subWork(f.value.fullName, 1);
						inFolder.begin(allMsg.size(), "Processing " + allMsg.size() + " bodies");
						for (List<RawImapBinding> slice : Lists.partition(allMsg, 1024)) {
							List<Long> itemIds = slice.stream().map(i -> i.itemId).toList();
							List<WithId<MailboxRecord>> items = recApi.slice(itemIds);
							for (var recWithId : items) {
								String bGuid = recWithId.value.messageBody;
								sdsStore.ifPresent(sds -> {
									if (repair && uploadIfMissing(mon, bodies, bGuid, sds)) {
										repaired.incrementAndGet();
										recWithId.value.flags.add(MailboxItemFlag.of("$placeholder", 0));
										recApi.updateById(recWithId.itemId, recWithId.value);
									}
								});
								inFolder.progress(1, "Processed " + bGuid);
							}
						}
						inFolder.end(true, "Finished for " + f.value.fullName, "{}");
					}
				}, ErrorCode.SQL_ERROR);
			}
			mon.end(true, "Repaired " + repaired.get(), "{}");
		}

		private boolean uploadIfMissing(RepairTaskMonitor mon, IDbMessageBodies bodies, String bGuid,
				ISdsSyncStore sds) {
			if (!sds.exists(ExistRequest.of(bGuid)).exists) {
				MessageBody loadedBody = bodies.get(bGuid);
				if (loadedBody != null) {
					byte[] fakeEml = eml(loadedBody);
					OffHeapTemporaryFile offheap = MemfdSupport.newOffHeapTemporaryFile(bGuid);
					try (OutputStream out = offheap.openForWriting()) {
						out.write(fakeEml);
					} catch (IOException e) {
						logger.error(e.getMessage(), e);
					}
					PutRequest pr = PutRequest.of(bGuid, offheap.path().toAbsolutePath().toString());
					sds.upload(pr);
					mon.log("Placeholder added for body {}.", bGuid);
					return true;
				}
			}
			return false;
		}

		private byte[] eml(MessageBody loadedBody) {
			Message msg = EmlBuilder.of(loadedBody, part -> {

				switch (part.mime) {
				case "text/plain":
					return new ByteArrayInputStream(loadedBody.preview.getBytes());
				case "text/html":
					String html = "<div>" + loadedBody.preview + "</div>";
					return new ByteArrayInputStream(html.getBytes());
				case "image/png":
					return MissingSdsBodiesRepair.class.getClassLoader().getResourceAsStream("repair-data/1px.png");
				case "image/jpg":
					return MissingSdsBodiesRepair.class.getClassLoader().getResourceAsStream("repair-data/1px.jpg");
				case "application/pdf":
					return MissingSdsBodiesRepair.class.getClassLoader().getResourceAsStream("repair-data/empty.pdf");
				default:
					return InputStream.nullInputStream();
				}
			});
			ByteArrayOutputStream out = new ByteArrayOutputStream();
			Mime4JHelper.serialize(msg, out);
			return out.toByteArray();
		}

	}

	private final BmContext context;

	public MissingSdsBodiesRepair(BmContext context) {
		this.context = context;
	}

	@Override
	public Set<MaintenanceOperation> availableOperations(Kind kind) {
		if (kind == Kind.USER || kind == Kind.MAILSHARE || kind == Kind.GROUP || kind == Kind.RESOURCE) {
			return Set.of(repairOp);
		} else {
			return Collections.emptySet();
		}
	}

	@Override
	public Set<InternalMaintenanceOperation> ops(Kind kind) {
		if (kind == Kind.USER || kind == Kind.MAILSHARE || kind == Kind.GROUP || kind == Kind.RESOURCE) {
			return Set.of(new UploadBodyPlaceholders(context));
		} else {
			return Collections.emptySet();
		}

	}
}
