package net.bluemind.backend.mail.replica.service.deferredaction;

import java.time.ZonedDateTime;
import java.util.Collections;
import java.util.List;
import java.util.Optional;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.backend.mail.api.IOutbox;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.deferredaction.api.DeferredAction;
import net.bluemind.deferredaction.api.IDeferredActionContainerUids;
import net.bluemind.deferredaction.api.IInternalDeferredAction;
import net.bluemind.deferredaction.registry.IDeferredActionExecutor;
import net.bluemind.domain.api.Domain;
import net.bluemind.domain.api.IInCoreDomains;

public class ScheduleMailDeferredActionExecutor implements IDeferredActionExecutor {

	private static final Logger logger = LoggerFactory.getLogger(ScheduleMailDeferredActionExecutor.class);

	private ServerSideServiceProvider provider;
	private IInCoreDomains domainsService;

	public ScheduleMailDeferredActionExecutor() {
		provider = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM);
		domainsService = provider.instance(IInCoreDomains.class);
	}

	@Override
	public void execute(ZonedDateTime executionDate) {
		domainsService.allUnfiltered().stream().filter(ScheduleMailDeferredActionExecutor::isNotGlobalVirt)
				.forEach(d -> executeForDomain(d, executionDate));
	}

	private void executeForDomain(ItemValue<Domain> domain, ZonedDateTime executionDate) {
		String deferredActionUid = IDeferredActionContainerUids.uidForDomain(domain.uid);
		IInternalDeferredAction deferredActionService = provider.instance(IInternalDeferredAction.class,
				deferredActionUid);

		List<ItemValue<DeferredAction>> deferredActions = deferredActionService
				.getByActionId(ScheduleMailDeferredAction.ACTION_ID, executionDate.toInstant().toEpochMilli());

		logger.debug("Found {} deferred actions of type {}", deferredActions.size(),
				ScheduleMailDeferredAction.ACTION_ID);

		deferredActions.stream().map(ScheduleMailDeferredActionExecutor::from) //
				.forEach(action -> {
					if (action.value.isPresent()) {
						flushOutbox(deferredActionService, ItemValue.create(action.uid, action.value.get()),
								domain.uid);
					} else {
						delete(deferredActionService, action);
					}
				});
	}

	private void delete(IInternalDeferredAction deferredActionService,
			ItemValue<Optional<ScheduleMailDeferredAction>> action) {

		try {
			deferredActionService.delete(action.uid);
		} catch (Exception e) {
			logger.error("Failed to delete {}", action.uid, e);
		}
	}

	private void flushOutbox(IInternalDeferredAction deferredActionService,
			ItemValue<ScheduleMailDeferredAction> action, String domainUid) {

		try {
			ServerSideServiceProvider userProvider = userProvider(action, domainUid);
			IOutbox outboxService = userProvider.instance(IOutbox.class, domainUid, action.value.mailboxUid);
			outboxService.flush();
		} catch (ServerFault e) {
			logger.error("Failed to flush deferred mail{}", action.uid, e);
		}
	}

	private ServerSideServiceProvider userProvider(ItemValue<ScheduleMailDeferredAction> action, String domainUid) {
		SecurityContext userSecurityContext = new SecurityContext(null, action.value.mailboxUid,
				Collections.emptyList(), Collections.emptyList(), Collections.emptyMap(), domainUid, "en",
				"ScheduleMailDeferredAction", false);
		ServerSideServiceProvider userProvider = ServerSideServiceProvider.getProvider(userSecurityContext);
		return userProvider;
	}

	static ItemValue<Optional<ScheduleMailDeferredAction>> from(ItemValue<DeferredAction> deferredAction) {
		try {
			ScheduleMailDeferredAction eventDeferredAction = new ScheduleMailDeferredAction(deferredAction.value);
			return ItemValue.create(deferredAction.uid, Optional.of(eventDeferredAction));
		} catch (Exception e) {
			return null;
		}

	}

	private static boolean isNotGlobalVirt(ItemValue<Domain> domain) {
		return !"global.virt".equals(domain.value.name);
	}

}
