/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2018
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.backend.mail.replica.service.internal.repair;

import java.util.Collection;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Multimap;
import com.google.common.collect.Sets;

import net.bluemind.backend.mail.api.IItemsTransfer;
import net.bluemind.backend.mail.api.MailboxFolder;
import net.bluemind.backend.mail.replica.api.IDbByContainerReplicatedMailboxes;
import net.bluemind.backend.mail.replica.api.IDbMailboxRecords;
import net.bluemind.backend.mail.replica.api.IMailReplicaUids;
import net.bluemind.backend.mail.replica.api.MailboxReplica;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.api.ContainerQuery;
import net.bluemind.core.container.api.IContainers;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.rest.BmContext;
import net.bluemind.directory.api.BaseDirEntry.Kind;
import net.bluemind.directory.api.MaintenanceOperation;
import net.bluemind.directory.service.IDirEntryRepairSupport;
import net.bluemind.directory.service.RepairTaskMonitor;
import net.bluemind.mailbox.api.Mailbox;

public class MultiInboxRepair implements IDirEntryRepairSupport {

	public static class RepairFactory implements IDirEntryRepairSupport.Factory {
		@Override
		public IDirEntryRepairSupport create(BmContext context) {
			return new MultiInboxRepair(context);
		}
	}

	public static final MaintenanceOperation minboxOp = MaintenanceOperation.create(IMailReplicaUids.REPAIR_MINBOX_OP,
			"Multiple INBOX in subtree");

	private static class FullNameDuplicatesMaintenance extends MailboxFoldersRepairOp {

		public FullNameDuplicatesMaintenance(BmContext ctx) {
			super(ctx, minboxOp.identifier, null, IMailReplicaUids.REPAIR_SUBTREE_OP, 1);
		}

		private static record Deduped(ItemValue<MailboxReplica> toKeep, List<ItemValue<MailboxReplica>> toTrash) {

		}

		@Override
		public void runOnFolders(boolean repair, RepairTaskMonitor monitor, String subTree, String domainUid,
				ItemValue<Mailbox> mbox, List<ItemValue<MailboxReplica>> fullList) {
			IDbByContainerReplicatedMailboxes foldersApi = context.provider()
					.instance(IDbByContainerReplicatedMailboxes.class, subTree);

			monitor.begin(1, "Inspecting subtree for mailbox " + mbox.value.name + "@" + domainUid);

			// get rid of orphan
			IContainers contApi = context.provider().instance(IContainers.class);
			Set<String> uniqueIdsOfRecords = contApi
					.allLight(ContainerQuery.ownerAndType(mbox.uid, IMailReplicaUids.MAILBOX_RECORDS)).stream()
					.map(c -> IMailReplicaUids.uniqueId(c.uid)).collect(Collectors.toSet());
			Sets.difference(uniqueIdsOfRecords, fullList.stream().map(iv -> iv.uid).collect(Collectors.toSet()))
					.forEach(uniqueId -> {
						monitor.log("Purging orphan records container with unique id " + uniqueId + "...");
						try {
							IDbMailboxRecords recsApi = context.provider().instance(IDbMailboxRecords.class, uniqueId);
							recsApi.prepareContainerDelete();
							contApi.delete(IMailReplicaUids.mboxRecords(uniqueId));
						} catch (ServerFault sf) {
							monitor.log("Skipped " + uniqueId + ", wrong datasource ?");
						}
					});

			Multimap<String, ItemValue<MailboxReplica>> byFullName = ArrayListMultimap.create();
			fullList.forEach(iv -> byFullName.put(iv.value.fullName, iv));
			List<Deduped> toPurge = new LinkedList<>();
			for (String fn : byFullName.keySet()) {
				Collection<ItemValue<MailboxReplica>> shouldBeOne = byFullName.get(fn);
				int len = shouldBeOne.size();
				if (len > 1) {
					// everything except the one with the biggest lastUid
					List<ItemValue<MailboxReplica>> bestFirst = shouldBeOne.stream()
							.sorted((i1, i2) -> Long.compare(i2.value.lastUid, i1.value.lastUid)).toList();
					Deduped dedup = new Deduped(bestFirst.getFirst(), bestFirst.subList(1, bestFirst.size()));
					toPurge.add(dedup);
				}
				monitor.progress(len, "process " + fn + " with " + (len - 1) + " duplicates");
			}
			for (Deduped itemValue : toPurge) {
				for (ItemValue<MailboxReplica> toTrash : itemValue.toTrash()) {
					monitor.log("Purge " + toTrash);
					monitor.notify("Obsolete replica item {}", toTrash.uid);
					if (repair) {
						IItemsTransfer forMerge = context.provider().instance(IItemsTransfer.class, toTrash.uid,
								itemValue.toKeep.uid);
						IDbMailboxRecords recsToKeep = context.provider().instance(IDbMailboxRecords.class,
								toTrash.uid);
						List<Long> allIds = recsToKeep.imapIdSet("1:*", "").stream().map(r -> r.itemId).toList();
						forMerge.move(allIds);
						foldersApi.delete(toTrash.uid);
					}
				}
			}

			if (mbox.value.type.sharedNs) {
				// foldersApi.delete might have trashed some messages and updated the Trash
				// folder with the parent_uid of the duplicated root
				ItemValue<MailboxFolder> root = foldersApi.root();
				ItemValue<MailboxReplica> trash = foldersApi.getCompleteById(foldersApi.trash().internalId);
				if (!trash.value.parentUid.equals(root.uid)) {
					monitor.notify("Fix {} parent -> {}", trash.value.fullName, root.uid);
					trash.value.parentUid = root.uid;
					foldersApi.update(trash.uid, trash.value);
				}

				ItemValue<MailboxReplica> sent = foldersApi.byReplicaName(mbox.value.name.replace('^', '.') + "/Sent");
				if (!sent.value.parentUid.equals(root.uid)) {
					monitor.notify("Fix {} parent -> {}", sent.value.fullName, root.uid);
					sent.value.parentUid = root.uid;
					foldersApi.update(sent.uid, sent.value);
				}
			}
		}

	}

	private final BmContext context;

	public MultiInboxRepair(BmContext context) {
		this.context = context;
	}

	@Override
	public Set<MaintenanceOperation> availableOperations(Kind kind) {
		if (kind == Kind.USER || kind == Kind.MAILSHARE || kind == Kind.GROUP || kind == Kind.RESOURCE) {
			return Set.of(minboxOp);
		} else {
			return Collections.emptySet();
		}
	}

	@Override
	public Set<InternalMaintenanceOperation> ops(Kind kind) {
		if (kind == Kind.USER || kind == Kind.MAILSHARE || kind == Kind.GROUP || kind == Kind.RESOURCE) {
			return Set.of(new FullNameDuplicatesMaintenance(context));
		} else {
			return Collections.emptySet();
		}

	}
}
