/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2022
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.backend.mail.replica.service.internal;

import java.sql.SQLException;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.backend.mail.replica.api.IDbMailboxRecords;
import net.bluemind.backend.mail.replica.api.IMailReplicaUids;
import net.bluemind.backend.mail.replica.api.IMailboxRecordExpunged;
import net.bluemind.backend.mail.replica.api.MailboxRecordExpunged;
import net.bluemind.backend.mail.repository.IMailboxRecordExpungedByFolderStore;
import net.bluemind.core.api.fault.ErrorCode;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.api.Count;
import net.bluemind.core.container.model.Container;
import net.bluemind.core.container.model.ContainerUid;
import net.bluemind.core.container.model.ItemFlagFilter;
import net.bluemind.core.container.repository.IContainerStore;
import net.bluemind.core.jdbc.JdbcAbstractStore;
import net.bluemind.core.rest.BmContext;
import net.bluemind.repository.provider.RepositoryProvider;

public class MailboxRecordExpungedService implements IMailboxRecordExpunged {

	private final BmContext context;
	private final IMailboxRecordExpungedByFolderStore expungedStore;
	private final Container folderContainer;

	private static final Logger logger = LoggerFactory.getLogger(MailboxRecordExpungedService.class);

	public MailboxRecordExpungedService(BmContext context, IMailboxRecordExpungedByFolderStore store,
			Container folderContainer) {
		this.context = context;
		this.expungedStore = store;
		this.folderContainer = folderContainer;
	}

	@Override
	public void delete(long itemId) {
		multipleDelete(List.of(itemId));
	}

	@Override
	public List<MailboxRecordExpunged> fetch() {
		try {
			return expungedStore.fetch();
		} catch (SQLException e) {
			throw ServerFault.sqlFault(e);
		}
	}

	@Override
	public Count count(ItemFlagFilter filter) {
		try {
			return Count.of(expungedStore.count());
		} catch (SQLException e) {
			throw ServerFault.sqlFault(e);
		}
	}

	@Override
	public MailboxRecordExpunged get(long itemId) {
		try {
			return expungedStore.get(itemId);
		} catch (SQLException e) {
			throw ServerFault.sqlFault(e);
		}
	}

	@Override
	public void multipleDelete(List<Long> itemIds) {
		JdbcAbstractStore.doOrFail(() -> {
			Map<Long, List<MailboxRecordExpunged>> imapUidsByContainerUid = itemIds.stream().map(this::get)
					.filter(Objects::nonNull).collect(Collectors.groupingBy(data -> data.containerId,
							Collectors.mapping(r -> r, Collectors.toList())));
			IContainerStore contStore = RepositoryProvider.instance(IContainerStore.class, context,
					ContainerUid.of(folderContainer.uid));
			for (Entry<Long, List<MailboxRecordExpunged>> entry : imapUidsByContainerUid.entrySet()) {
				Long containerId = entry.getKey();
				List<MailboxRecordExpunged> imapUids = entry.getValue();
				Container cont = contStore.get(containerId);
				try {
					context.provider().instance(IDbMailboxRecords.class, IMailReplicaUids.uniqueId(cont.uid))
							.deleteImapUids(imapUids.stream().map(mr -> mr.imapUid).toList());
					logger.info("Purge expunged message {} of queue for container {}", imapUids, containerId);
					imapUids.forEach(mr -> ServerFault.onExceptionVoid(() -> expungedStore.delete(mr.itemId),
							ErrorCode.SQL_ERROR));
				} catch (Exception e) {
					logger.error("Error cleaning up message {} on container {}: {}", imapUids, containerId,
							e.getMessage());
				}
			}

			return null;
		});

	}
}
