/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2024
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.backend.mail.replica.service.internal;

import java.sql.SQLException;
import java.util.Optional;

import javax.sql.DataSource;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.backend.mail.replica.api.IDbMailboxRecords;
import net.bluemind.backend.mail.replica.api.IInternalMailboxesByLocation;
import net.bluemind.backend.mail.replica.api.MailboxReplica;
import net.bluemind.backend.mail.replica.persistence.MailboxReplicaStore;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.Container;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.container.persistence.ContainerStore;
import net.bluemind.core.container.service.internal.ContainerStoreService;
import net.bluemind.core.container.service.internal.RBACManager;
import net.bluemind.core.rest.BmContext;
import net.bluemind.role.api.BasicRoles;

public class MailboxesByLocationService implements IInternalMailboxesByLocation {

	private static final Logger logger = LoggerFactory.getLogger(MailboxesByLocationService.class);

	private final BmContext context;
	private final String dataLocation;
	private final DataSource dataSource;

	public MailboxesByLocationService(BmContext context, String dataLocation) {
		this.context = context;
		this.dataLocation = dataLocation;
		this.dataSource = context.getMailboxDataSource(dataLocation);
	}

	@Override
	public void deleteMailbox(String uid) {
		RBACManager.forContext(context).check(BasicRoles.ROLE_ADMIN);

		try {
			ContainerStore containerStore = new ContainerStore(context, dataSource, context.getSecurityContext());
			Container container = containerStore.get(uid);
			if (container == null) {
				throw ServerFault.notFound("Container " + uid + " is missing.");
			}

			MailboxReplicaStore mboxReplicaStore = new MailboxReplicaStore(dataSource, container, container.domainUid);
			ContainerStoreService<MailboxReplica> storeService = new ContainerStoreService<>(context, container,
					mboxReplicaStore);

			IDbMailboxRecords recordsApi = null;
			try {
				recordsApi = context.provider().instance(IDbMailboxRecords.class, container.uid);
				recordsApi.prepareContainerDelete();
			} catch (ServerFault sf) {
				logger.warn("Records API does not work for {}: {}", container.uid, sf.getMessage());
			}

			ItemValue<MailboxReplica> replicaToDelete = getCompleteReplica(container.uid, storeService);
			logger.warn("***** Will delete {}", replicaToDelete);

			DbReplicatedMailboxesService.deleteMailbox(context, uid, storeService, recordsApi, container);

		} catch (SQLException e) {
			throw ServerFault.sqlFault(e);
		}

	}

	private ItemValue<MailboxReplica> getCompleteReplica(String uid,
			ContainerStoreService<MailboxReplica> storeService) {
		return Optional.ofNullable(MboxReplicasCache.byUid(uid)).orElseGet(() -> {
			ItemValue<MailboxReplica> fetched = storeService.get(uid, null);
			if (fetched != null) {
				fetched.value.dataLocation = dataLocation;
				MboxReplicasCache.cache(fetched);
			}
			return fetched;
		});
	}
}
