/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2017
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.backend.mail.replica.service.internal;

import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.Lists;

import net.bluemind.backend.mail.api.FolderCounters;
import net.bluemind.backend.mail.api.IItemsTransfer;
import net.bluemind.backend.mail.api.IMailboxFolders;
import net.bluemind.backend.mail.api.IMailboxFoldersByContainer;
import net.bluemind.backend.mail.api.IMailboxFoldersByOwner;
import net.bluemind.backend.mail.api.ImportMailboxItemSet;
import net.bluemind.backend.mail.api.ImportMailboxItemsStatus;
import net.bluemind.backend.mail.api.MailboxFolder;
import net.bluemind.backend.mail.api.flags.MailboxItemFlag;
import net.bluemind.backend.mail.api.utils.FolderTree;
import net.bluemind.backend.mail.replica.api.IDbByContainerReplicatedMailboxes;
import net.bluemind.backend.mail.replica.api.IDbMailboxRecords;
import net.bluemind.backend.mail.replica.api.IMailReplicaUids;
import net.bluemind.backend.mail.replica.api.MailboxRecord;
import net.bluemind.backend.mail.replica.api.MailboxRecord.InternalFlag;
import net.bluemind.backend.mail.replica.api.MailboxReplica;
import net.bluemind.backend.mail.replica.api.MailboxReplicaRootDescriptor;
import net.bluemind.backend.mail.replica.api.WithId;
import net.bluemind.backend.mail.repository.IMailboxReplicaStore;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.api.Ack;
import net.bluemind.core.container.api.IOfflineMgmt;
import net.bluemind.core.container.api.IdRange;
import net.bluemind.core.container.model.Container;
import net.bluemind.core.container.model.ItemIdentifier;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.container.model.ItemVersion;
import net.bluemind.core.container.model.acl.Verb;
import net.bluemind.core.container.repository.IAclStore;
import net.bluemind.core.container.repository.IContainerStore;
import net.bluemind.core.container.service.internal.ContainerStoreService;
import net.bluemind.core.container.service.internal.RBACManager;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.directory.api.DirEntry;
import net.bluemind.directory.api.IDirectory;

public class ImapReplicatedMailboxesService extends BaseReplicatedMailboxesService
		implements IMailboxFolders, IMailboxFoldersByContainer, IMailboxFoldersByOwner {

	private static final Logger logger = LoggerFactory.getLogger(ImapReplicatedMailboxesService.class);

	public ImapReplicatedMailboxesService(MailboxReplicaRootDescriptor root, Container cont, BmContext context,
			IMailboxReplicaStore store, ContainerStoreService<MailboxReplica> mboxReplicaStore,
			IContainerStore contStore, IAclStore aclStore) {
		super(root, cont, context, store, mboxReplicaStore, contStore, aclStore);
	}

	@Override
	public ItemValue<MailboxFolder> getCompleteById(long id) {
		rbac.check(Verb.Visible.name());
		return adapt(storeService.get(id, null));
	}

	@Override
	public List<ItemValue<MailboxFolder>> multipleGetById(List<Long> ids) {
		rbac.check(Verb.Visible.name());
		return storeService.getMultipleById(ids).stream().map(this::adapt).collect(Collectors.toList());
	}

	@Override
	public Ack updateById(long id, MailboxFolder v) {
		rbac.check(Verb.Write.name());

		ItemValue<MailboxFolder> current = getCompleteById(id);
		if (current == null) {
			throw ServerFault.notFound("mailboxReplica with id " + id + " not found.");
		}

		MailboxFolder value = nameSanitizer.sanitizeNames(v);

		if (value.fullName.equals(current.value.fullName)) {
			logger.warn("Rename attempt to same name '{}'", value.fullName);
			ItemVersion touchVersion = storeService.touch(current.uid);
			return touchVersion.ack();
		}

		FolderTree fullTree = FolderTree.of(all());
		List<ItemValue<MailboxFolder>> renameTargets = fullTree.children(current);
		IDbByContainerReplicatedMailboxes writeDelegate = context.provider()
				.instance(IDbByContainerReplicatedMailboxes.class, container.uid);
		ItemValue<MailboxReplica> curReplica = writeDelegate.getCompleteById(current.internalId);
		MailboxReplica curCopy = MailboxReplica.from(curReplica.value);
		curCopy.name = null;
		curCopy.parentUid = null;
		curCopy.fullName = value.fullName;
		Ack lastUpd = writeDelegate.update(curReplica.uid, curCopy);
		for (ItemValue<MailboxFolder> tgt : renameTargets) {
			ItemValue<MailboxReplica> parent = getCompleteReplica(tgt.value.parentUid);
			ItemValue<MailboxReplica> replicaTgt = getCompleteReplica(tgt.uid);
			MailboxReplica tgtCopy = MailboxReplica.from(replicaTgt.value);
			tgtCopy.name = null;
			tgtCopy.parentUid = null;
			tgtCopy.fullName = parent.value.fullName + "/" + replicaTgt.value.name;
			lastUpd = writeDelegate.update(tgt.uid, tgtCopy);
		}
		return lastUpd;
	}

	@Override
	public ItemIdentifier createForHierarchy(long hierId, MailboxFolder v) {
		rbac.check(Verb.Write.name());

		MailboxFolder value = nameSanitizer.sanitizeNames(v);

		ItemValue<MailboxReplica> folder = byReplicaName(value.fullName);
		if (folder != null) {
			return ItemIdentifier.of(folder.uid, folder.internalId, folder.version, folder.timestamp());
		}

		FolderInternalIdCache.storeExpectedRecordId(container, value.fullName, hierId);
		IDbByContainerReplicatedMailboxes writeDelegate = context.provider()
				.instance(IDbByContainerReplicatedMailboxes.class, container.uid);
		MailboxReplica mr = new MailboxReplica();
		mr.fullName = value.fullName;
		mr.lastUid = 0;
		mr.uidValidity = 0;
		mr.lastAppendDate = new Date();
		mr.deleted = false;
		String pipoUid = UUID.randomUUID().toString();
		writeDelegate.create(pipoUid, mr);
		ItemValue<MailboxReplica> created = getCompleteReplica(pipoUid);
		return ItemIdentifier.of(created.uid, created.internalId, created.version, created.timestamp());
	}

	@Override
	public ItemIdentifier createBasic(MailboxFolder value) {
		rbac.check(Verb.Write.name());

		IOfflineMgmt offlineApi = context.provider().instance(IOfflineMgmt.class, container.domainUid, container.owner);
		IdRange alloc = offlineApi.allocateOfflineIds(1);
		return createForHierarchy(alloc.globalCounter, value);
	}

	private static final Set<String> USER_PROTECTED = Set.of("INBOX", "Sent", "Drafts", "Trash", "Junk", "Outbox",
			"Templates");

	@Override
	public void deleteById(long id) {
		rbac.check(Verb.Write.name());

		ItemValue<MailboxFolder> toDelete = getCompleteById(id);
		if (toDelete == null || toDelete.value == null) {
			throw ServerFault.notFound("Folder with id " + id + " not found");
		}
		if (toDelete.value.deleted) {
			throw ServerFault.notFound("Folder with id " + id + " has already been deleted.");
		}

		if (toDelete.value.parentUid == null && USER_PROTECTED.contains(toDelete.value.name)) {
			throw new ServerFault("Deletion of " + id + " prevented.");
		}

		IDbByContainerReplicatedMailboxes writeDelegate = context.provider()
				.instance(IDbByContainerReplicatedMailboxes.class, container.uid);
		writeDelegate.delete(toDelete.uid);
	}

	private void notifyUpdate(ItemValue<MailboxFolder> parent, long version) {
		EmitReplicationEvents.subtreeUpdated(container.uid, container.owner,
				ItemIdentifier.of(parent.uid, parent.internalId, version, parent.timestamp()), false);
	}

	@Override
	public void deepDelete(long id) {
		rbac.check(Verb.Write.name());

		ItemValue<MailboxFolder> toDelete = getCompleteById(id);
		if (toDelete == null || toDelete.value == null) {
			throw ServerFault.notFound("Folder with id " + id + " not found");
		}
		logger.info("Start deepDelete of {}...", toDelete);
		deleteChildFolders(toDelete);
		deleteById(id);
	}

	public void emptyFolder(long id) {
		rbac.check(Verb.Write.name());
		emptyFolder(id, true);
	}

	public void removeMessages(long id) {
		rbac.check(Verb.Write.name());
		emptyFolder(id, false);
	}

	protected void emptyFolder(long id, boolean deleteChildFolders) {
		ItemValue<MailboxFolder> folder = getCompleteById(id);
		if (folder == null || folder.value == null) {
			throw ServerFault.notFound("Folder with id " + id + " not found");
		}
		IDbMailboxRecords recordService = context.provider().instance(IDbMailboxRecords.class, folder.uid);
		logger.info("Start emptying {} (deleteChildFolders={})...", folder, deleteChildFolders);
		if (deleteChildFolders) {
			deleteChildFolders(folder);
		}
		logger.info("On purge of '{}'", folder.value.fullName);
		if (folder.value.name.equals("Trash") && folder.value.parentUid == null) {
			flag(folder, MailboxItemFlag.System.Deleted);
		} else {
			DirEntry dirEntry = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM)
					.instance(IDirectory.class, container.domainUid).findByEntryUid(container.owner);
			String subtreeContainer = IMailReplicaUids.subtreeUid(container.domainUid, dirEntry);
			Trash trash = new Trash(context, subtreeContainer, recordService);
			trash.deleteItems(folder.internalId, recordService.all().stream().map(rec -> rec.internalId).toList());
		}
	}

	public void markFolderAsRead(long id) {
		rbac.check(Verb.Write.name());
		ItemValue<MailboxFolder> folder = getCompleteById(id);
		if (folder == null || folder.value == null) {
			throw ServerFault.notFound("Folder with id " + id + " not found");
		}
		flag(folder, MailboxItemFlag.System.Seen);
	}

	protected void flag(ItemValue<MailboxFolder> folder, MailboxItemFlag.System flag) {
		IDbMailboxRecords writeDelegate = context.provider().instance(IDbMailboxRecords.class, folder.uid);
		List<Long> allIds = writeDelegate.imapIdSet("1:*", "").stream().map(r -> r.itemId).toList();
		for (List<Long> slice : Lists.partition(allIds, 500)) {
			List<WithId<MailboxRecord>> recSlice = writeDelegate.slice(slice);
			List<MailboxRecord> updates = recSlice.stream().map(wid -> {
				MailboxRecord mr = wid.value;
				mr.flags.add(flag.value());
				if (flag == MailboxItemFlag.System.Deleted) {
					mr.internalFlags.add(InternalFlag.expunged);
				}
				return mr;
			}).toList();
			writeDelegate.updates(updates);
		}
		if (allIds.isEmpty()) {
			logger.info("No item to mark as {} in folder {}", flag, folder);
		} else {
			ItemVersion touched = storeService.touch(folder.uid);
			notifyUpdate(folder, touched.version);
		}
	}

	private void deleteChildFolders(ItemValue<MailboxFolder> toClean) {
		FolderTree fullTree = FolderTree.of(childrensOf(toClean.uid));
		List<ItemValue<MailboxFolder>> children = fullTree.children(toClean);
		Collections.reverse(children);
		children.forEach(child -> deleteById(child.internalId));
	}

	@Override
	public ImportMailboxItemsStatus importItems(long id, ImportMailboxItemSet mailboxItems) throws ServerFault {
		ItemValue<MailboxFolder> destinationFolder = getCompleteById(id);
		if (destinationFolder == null) {
			throw new ServerFault("Cannot find destination mailboxfolder");
		}

		if (!rbac.can(Verb.Write.name())) {
			RBACManager destFolder = RBACManager.forContext(context)
					.forContainer(IMailReplicaUids.mboxRecords(destinationFolder.uid));
			destFolder.check(Verb.Write.name());
		}

		ItemValue<MailboxFolder> sourceFolder = getCompleteById(mailboxItems.mailboxFolderId);
		if (sourceFolder == null) {
			throw new ServerFault("Cannot find source mailboxfolder");
		}

		logger.info("[{}] Moving {} item(s) from {} into {}", container.name, mailboxItems.ids.size(),
				sourceFolder.value.fullName, destinationFolder.value.fullName);

		IItemsTransfer transferApi = context.provider().instance(IItemsTransfer.class, sourceFolder.uid,
				destinationFolder.uid);
		boolean move = mailboxItems.deleteFromSource;
		List<Long> sourceObjects = mailboxItems.ids.stream().map(k -> k.id).toList();
		List<ItemIdentifier> copies = move ? transferApi.move(sourceObjects) : transferApi.copy(sourceObjects);
		return ImportMailboxItemsStatus.fromTransferResult(sourceObjects, copies);

	}

	@Override
	public List<FolderCounters> counters(List<Long> folderItemIds) {
		return super.folderCounters(folderItemIds);
	}

}
