/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of Blue Mind. Blue Mind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License)
  * or the CeCILL as published by CeCILL.info (version 2 of the License).
  *
  * There are special exceptions to the terms and conditions of the
  * licenses as they are applied to this program. See LICENSE.txt in
  * the directory of this program distribution.
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.backend.mail.replica.persistence;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Timestamp;
import java.time.Instant;
import java.util.Optional;

import javax.sql.DataSource;

import io.netty.buffer.ByteBufUtil;
import net.bluemind.backend.mail.repository.IMessageBodyPurgeQueueStore;
import net.bluemind.core.jdbc.JdbcAbstractStore;

public class MessageBodyPurgeQueueStore extends JdbcAbstractStore implements IMessageBodyPurgeQueueStore {

	public MessageBodyPurgeQueueStore(DataSource dataSource) {
		super(dataSource);
	}

	private static final String INSERTQUERY = """
			INSERT INTO t_message_body_purge_queue (message_body_guid, created, removed, immediate_remove)
			VALUES (?, ?, ?, ?)
			ON CONFLICT (message_body_guid)
			DO UPDATE SET created=EXCLUDED.created, removed=EXCLUDED.removed, immediate_remove=EXCLUDED.immediate_remove""";

	@Override
	public boolean createOrUpdate(String guid, Instant created, Instant removed, boolean immediateRemove)
			throws SQLException {

		try (Connection con = getConnection();
				Statement st = con.createStatement();
				PreparedStatement ist = con.prepareStatement(INSERTQUERY)) {
			con.setAutoCommit(false);
			st.execute("SET bluemind.bypass_message_body_purge_queue = false");
			ist.setBytes(1, ByteBufUtil.decodeHexDump(guid));
			ist.setTimestamp(2, Timestamp.from(created));
			ist.setTimestamp(3, removed != null ? Timestamp.from(removed) : null);
			ist.setBoolean(4, immediateRemove);
			int updated = ist.executeUpdate();
			st.execute("RESET bluemind.bypass_message_body_purge_queue");
			con.commit();
			return updated > 0;
		}
	}

	private static final String DELETEQUERY = "DELETE FROM t_message_body_purge_queue WHERE message_body_guid = ?";

	@Override
	public long delete(String guid) throws SQLException {
		try (Connection con = getConnection();
				Statement st = con.createStatement();
				PreparedStatement dst = con.prepareStatement(DELETEQUERY)) {
			con.setAutoCommit(false);
			st.execute("SET bluemind.bypass_message_body_purge_queue = false");
			dst.setBytes(1, ByteBufUtil.decodeHexDump(guid));
			int removed = dst.executeUpdate();
			st.execute("RESET bluemind.bypass_message_body_purge_queue");
			con.commit();
			return removed;
		}
	}

	@Override
	public void deleteAll() throws SQLException {
		update("TRUNCATE t_message_body_purge_queue", new Object[] {});
	}

	@Override
	public BodyPurge get(String guid) throws SQLException {
		return unique("""
				SELECT encode(message_body_guid, 'hex'), created, removed, immediate_remove
				FROM t_message_body_purge_queue
				WHERE message_body_guid = ?""", //
				rs -> new BodyPurge( //
						rs.getString(1), //
						Optional.ofNullable(rs.getTimestamp(2)).map(Timestamp::toInstant).orElse(null), //
						Optional.ofNullable(rs.getTimestamp(3)).map(Timestamp::toInstant).orElse(null), //
						rs.getBoolean(4)), //
				(rs, index, v) -> index, new Object[] { ByteBufUtil.decodeHexDump(guid) });
	}

	@Override
	public void enableReplicationTriggers() throws SQLException {
		select("SELECT fn_enable_message_body_purge_queue_sync_triggers()", rs -> true, (rs, index, value) -> index,
				new Object[] {});
		update("ALTER SYSTEM SET bluemind.bypass_message_body_purge_queue = false", new Object[] {});
		select("select pg_reload_conf()", rs -> true, (rs, index, value) -> index, new Object[] {});
	}

	@Override
	public void disableReplicationTriggers() throws SQLException {
		select("SELECT fn_disable_message_body_purge_queue_sync_triggers()", rs -> true, (rs, index, value) -> index,
				new Object[] {});
		update("ALTER SYSTEM SET bluemind.bypass_message_body_purge_queue = true", new Object[] {});
		select("select pg_reload_conf()", rs -> true, (rs, index, value) -> index, new Object[] {});
	}

	@Override
	public void resyncToKafka() throws SQLException {
		try (Connection conn = getConnection()) {
			conn.setTransactionIsolation(Connection.TRANSACTION_SERIALIZABLE);
			conn.setAutoCommit(false);
			try (Statement stmt = conn.createStatement()) {
				stmt.execute("""
						INSERT INTO tx_outbox_for_kafka(domain_uid, part_key, kafka_key, kafka_value)
						    VALUES (
						        'sync.q',
						        'pq',
						        convert_to(
						            json_build_object(
						                'type', 'pq',
						                'owner', '',
						                'uid', '',
						                'id', -1,
						                'valueClass', '',
						                'operation', 'TRUNCATE'
						            )::text, 'UTF-8'
						        ),
						        convert_to(
						            json_build_object(
						                'operation', 'TRUNCATE',
						                'table', 't_message_body_purge_queue'
						            )::text, 'UTF-8'
						        )
						    )""");
				stmt.execute("""
						INSERT INTO tx_outbox_for_kafka(domain_uid, part_key, kafka_key, kafka_value)
						SELECT
						    'sync.q',
						    'pq',
						    convert_to(
						        json_build_object(
						            'type', 'pq',
						            'owner', '',
						            'id', -1,
						            'uid', encode(n.message_body_guid, 'hex'),
						            'valueClass', '',
						            'operation', 'SYNC'
						        )::text, 'UTF-8'
						    ),
						    convert_to(
						        json_build_object(
						            'created', (EXTRACT(EPOCH FROM n.created) * 1000)::bigint,
						            'removed', (EXTRACT(EPOCH FROM n.removed) * 1000)::bigint,
						            'immediate_remove', n.immediate_remove
						        )::text, 'UTF-8'
						    )
						FROM t_message_body_purge_queue n""");
				conn.commit();
			}
		}
	}
}
