/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2022
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.backend.mail.replica.persistence;

import java.sql.SQLException;
import java.util.Arrays;
import java.util.List;

import javax.sql.DataSource;

import net.bluemind.backend.mail.replica.api.MailboxRecordExpunged;
import net.bluemind.backend.mail.repository.IMailboxRecordExpungedByFolderStore;
import net.bluemind.core.container.model.Container;
import net.bluemind.core.jdbc.JdbcAbstractStore;

public class MailboxRecordExpungedByFolderStore extends JdbcAbstractStore
		implements IMailboxRecordExpungedByFolderStore {

	protected static final Creator<MailboxRecordExpunged> MB_CREATOR = rs -> new MailboxRecordExpunged();

	public static final Integer LIMIT = 1000;

	private Container folderContainer;
	private Container subtreeContainer;

	public MailboxRecordExpungedByFolderStore(DataSource pool, Container folderContainer, Container subtreeContainer) {
		super(pool);
		this.folderContainer = folderContainer;
		this.subtreeContainer = subtreeContainer;
	}

	private static final String CREATE_QUERY = "INSERT INTO q_mailbox_record_expunged ( "
			+ MailboxRecordExpungedColumns.COLUMNS.names() + ") VALUES ("
			+ MailboxRecordExpungedColumns.COLUMNS.values() + " ON CONFLICT DO NOTHING";

	public void store(MailboxRecordExpunged value) throws SQLException {
		insert(CREATE_QUERY, value, Arrays.asList(MailboxRecordExpungedColumns.values(value.itemId),
				MailboxRecordExpungedColumns.values(null)));
	}

	@Override
	public void delete(Long itemId) throws SQLException {
		delete("DELETE FROM q_mailbox_record_expunged WHERE container_id = ? AND subtree_id = ? AND item_id = ?",
				new Object[] { folderContainer.id, subtreeContainer.id, itemId });
	}

	private static final String GET_QUERY = "SELECT " + MailboxRecordExpungedColumns.COLUMNS.names()
			+ " FROM q_mailbox_record_expunged WHERE container_id = ? AND subtree_id = ? AND item_id = ?";

	@Override
	public MailboxRecordExpunged get(Long itemId) throws SQLException {
		return unique(GET_QUERY, MB_CREATOR, MailboxRecordExpungedColumns.populator(itemId),
				new Object[] { folderContainer.id, subtreeContainer.id, itemId });
	}

	private static final String FETCH_QUERY = "SELECT " + MailboxRecordExpungedColumns.COLUMNS.names()
			+ " FROM q_mailbox_record_expunged WHERE container_id = ? AND subtree_id = ?";

	@Override
	public List<MailboxRecordExpunged> fetch() throws SQLException {
		return select(FETCH_QUERY, con -> new MailboxRecordExpunged(), POPULATOR,
				new Object[] { folderContainer.id, subtreeContainer.id });
	}

	private static final EntityPopulator<MailboxRecordExpunged> POPULATOR = (rs, index, value) -> {
		value.containerId = (long) rs.getInt(index++);
		value.subtreeId = (long) rs.getInt(index++);
		value.itemId = rs.getLong(index++);
		value.imapUid = rs.getLong(index++);
		value.created = rs.getDate(index++);
		return index;
	};

	@Override
	public Long count() throws SQLException {
		String q = "SELECT COUNT(*) FROM q_mailbox_record_expunged WHERE container_id = ? AND subtree_id = ?";
		return unique(q, rs -> rs.getLong(1), (rs, index, v) -> index,
				new Object[] { folderContainer.id, subtreeContainer.id });
	}

}
