/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2017
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.backend.mail.replica.persistence;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Timestamp;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;

import javax.sql.DataSource;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.backend.mail.api.MessageBody;
import net.bluemind.backend.mail.repository.IMessageBodyStore;
import net.bluemind.core.container.persistence.BytesCreator;
import net.bluemind.core.container.persistence.StringCreator;
import net.bluemind.core.jdbc.JdbcAbstractStore;
import net.bluemind.utils.ProgressPrinter;

public class MessageBodyStore extends JdbcAbstractStore implements IMessageBodyStore {

	private static final Logger logger = LoggerFactory.getLogger(MessageBodyStore.class);

	private static final Creator<MessageBody> MB_CREATOR = rs -> new MessageBody();

	private static final int DELETE_ORPHAN_BATCHSIZE = 10_000;

	public MessageBodyStore(DataSource pool) {
		super(pool);
		Objects.requireNonNull(pool, "datasource must not be null");
	}

	private static final String CREATE_OR_UPDATE_QUERY = """
			    INSERT INTO t_message_body (
			        %s, guid
			    ) VALUES (
			        %s, decode(?, 'hex')
			    ) ON CONFLICT (guid) DO UPDATE SET (
			        %s
			    ) = (
			        %s
			    ) WHERE (
			        %s
			    ) IS DISTINCT FROM (
			        %s
			    )
			""".formatted( //
			MessageBodyColumns.COLUMNS.names(), //
			MessageBodyColumns.COLUMNS.values(), //
			MessageBodyColumns.COLUMNS.names(), //
			MessageBodyColumns.COLUMNS.values(), //
			MessageBodyColumns.COLUMNS.names("t_message_body"), //
			MessageBodyColumns.COLUMNS.names("EXCLUDED"));

	public int store(MessageBody value) throws SQLException {
		return insert(CREATE_OR_UPDATE_QUERY, value,
				Arrays.asList(MessageBodyColumns.values(value.guid), MessageBodyColumns.values(null)));
	}

	private static final String GET_QUERY = "SELECT " + MessageBodyColumns.COLUMNS.names()
			+ " FROM t_message_body WHERE guid = decode(?, 'hex')";

	public MessageBody get(String guid) throws SQLException {
		return unique(GET_QUERY, MB_CREATOR, MessageBodyColumns.populator(guid), new Object[] { guid });
	}

	private static final String MGET_QUERY = "SELECT encode(guid, 'hex'), " + MessageBodyColumns.COLUMNS.names()
			+ " FROM t_message_body WHERE guid = ANY(?::bytea[])";

	public List<MessageBody> multiple(String... guids) throws SQLException {
		return multiple(new Object[] { toByteArray(guids) });
	}

	public List<MessageBody> multiple(List<String> guids) throws SQLException {
		return multiple(new Object[] { toByteArray(guids) });
	}

	private List<MessageBody> multiple(Object[] byteArrays) throws SQLException {
		return select(MGET_QUERY, MB_CREATOR, (rs, index, mb) -> {
			mb.guid = rs.getString(index++);
			return MessageBodyColumns.simplePopulator().populate(rs, index, mb);
		}, byteArrays);
	}

	public boolean exists(String uid) throws SQLException {
		String q = "SELECT 1 FROM t_message_body WHERE guid = decode(?, 'hex')";
		Boolean found = unique(q, rs -> Boolean.TRUE, Collections.emptyList(), new Object[] { uid });
		return found != null;
	}

	public List<String> existing(List<String> toCheck) throws SQLException {
		List<String> theList = java.util.Optional.ofNullable(toCheck).orElse(Collections.emptyList());
		String q = "SELECT encode(guid, 'hex') FROM t_message_body WHERE guid = ANY(?::bytea[])";
		return select(q, StringCreator.FIRST, (rs, index, val) -> index, new Object[] { toByteArray(theList) });
	}

	public List<String> deleteOrphanBodies() throws SQLException {
		String query = """
				DELETE FROM t_message_body mb
				USING t_message_body_purge_queue pq
				WHERE pq.message_body_guid = mb.guid
				AND pq.created <= now() - '2 hours'::interval
				RETURNING encode(mb.guid, 'hex')""";
		List<String> selected = delete(query, StringCreator.FIRST, Arrays.asList((rs, index, val) -> index));
		int size = selected.size();
		logger.info("{} orphan bodies purged.", size);
		if (size > 0) {
			markPurgeQueueRemoved();
		}
		return selected;
	}

	/**
	 * Mark any entry of t_message_body_purge_queue which are not present inside
	 * t_message_body for later removal. The removal is done by an external
	 * verticle, which uses a configurable remove delay in SystemConfiguration.
	 *
	 * @throws SQLException
	 */
	private void markPurgeQueueRemoved() throws SQLException {
		String query = """
				UPDATE t_message_body_purge_queue SET removed=now() FROM (
				    SELECT pq.message_body_guid
				    FROM t_message_body_purge_queue pq
				    LEFT JOIN t_message_body mb ON (mb.guid = pq.message_body_guid)
				    WHERE mb.guid IS NULL
				) pqnull
				WHERE removed IS NULL
				AND pqnull.message_body_guid = t_message_body_purge_queue.message_body_guid""";
		update(query, null);
	}

	public List<String> deletePurgedBodies(Instant removedBefore, long limit) throws SQLException {
		String query = """
				WITH bodies AS (
				    SELECT message_body_guid
				    FROM t_message_body_purge_queue
				    WHERE removed IS NOT NULL AND (
				        removed <= ? OR immediate_remove IS true
				    )
				    LIMIT ?
				 )
				 DELETE FROM t_message_body_purge_queue
				 WHERE message_body_guid IN (
				     SELECT message_body_guid FROM bodies
				 ) RETURNING encode(message_body_guid, 'hex')""";
		return delete(query, StringCreator.FIRST, Arrays.asList((rs, index, val) -> index),
				new Object[] { Timestamp.from(removedBefore), limit });
	}

	private String[] toByteArray(String... guids) {
		int len = guids.length;
		String[] ret = new String[len];
		for (int i = 0; i < len; i++) {
			ret[i] = "\\x" + guids[i];
		}
		return ret;
	}

	private String[] toByteArray(List<String> guids) {
		String[] ret = new String[guids.size()];
		int i = 0;
		for (String guid : guids) {
			ret[i++] = "\\x" + guid;
		}
		return ret;
	}

	@Override
	public void deleteNoRecordNoPurgeOrphanBodies() throws SQLException {
		try ( //
				Connection connexion = this.datasource.getConnection(); //
				Statement setupStmt = connexion.createStatement();

				PreparedStatement selectBodyStmt = connexion.prepareStatement("""
						SELECT t_message_body.guid FROM t_message_body
						LEFT JOIN t_mailbox_record ON t_message_body.guid = t_mailbox_record.message_body_guid
						LEFT JOIN t_message_body_purge_queue AS q ON t_message_body.guid = q.message_body_guid
						WHERE t_mailbox_record.message_body_guid IS NULL
						AND q.message_body_guid IS NULL"""); //
				Connection writeConnexion = this.datasource.getConnection(); //
				PreparedStatement deleteBodyStmt = writeConnexion.prepareStatement("""
						INSERT INTO t_message_body_purge_queue (message_body_guid, created, immediate_remove)
						VALUES (?, now() - '1 hour'::interval, true)
						ON CONFLICT(message_body_guid) DO NOTHING""");//
		) {
			setupStmt.execute("set max_parallel_workers_per_gather=8");

			selectBodyStmt.setFetchSize(DELETE_ORPHAN_BATCHSIZE); // NOSONAR: OK
			writeConnexion.setAutoCommit(false);

			try (ResultSet selectBodyRs = selectBodyStmt.executeQuery()) {
				ProgressPrinter progress = new ProgressPrinter(0);
				long i = 0;
				while (selectBodyRs.next()) {
					byte[] guid = selectBodyRs.getBytes(1);
					deleteBodyStmt.setBytes(1, guid);
					progress.add();
					deleteBodyStmt.addBatch();
					if (i % DELETE_ORPHAN_BATCHSIZE == 0) {
						deleteBodyStmt.executeBatch();
						writeConnexion.commit();
					}
					if (progress.shouldPrint()) {
						logger.info("Deleting orphan bodies progress: {}", progress.toString());
					}
					i++;
				}
				deleteBodyStmt.executeBatch();
				writeConnexion.commit();
			}
		}

		String selectOrphans = """
				SELECT t_message_body.guid FROM t_message_body
				LEFT JOIN t_mailbox_record ON t_message_body.guid = t_mailbox_record.message_body_guid
				LEFT JOIN t_message_body_purge_queue AS q ON t_message_body.guid = q.message_body_guid
				WHERE t_mailbox_record.message_body_guid IS NULL
				AND q.message_body_guid IS NULL""";
		String deleteOrphan = """
				INSERT INTO t_message_body_purge_queue (message_body_guid, created, immediate_remove)
				VALUES (?, now() - '1 hour'::interval, true)
				ON CONFLICT(message_body_guid) DO NOTHING""";

		List<byte[]> orphanGuids = select(selectOrphans, BytesCreator.FIRST, (rs, index, val) -> index,
				new Object[] {});
		for (byte[] orphanGuid : orphanGuids) {
			insert(deleteOrphan, new Object[] { orphanGuid });
		}
	}

}
