package net.bluemind.backend.mail.partfile;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.RandomAccessFile;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel.MapMode;
import java.nio.file.Files;
import java.nio.file.Path;
import java.time.Duration;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.function.Supplier;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Stopwatch;
import com.google.common.base.Suppliers;

import io.lettuce.core.RedisClient;
import io.lettuce.core.api.StatefulRedisConnection;
import io.lettuce.core.api.async.RedisAsyncCommands;
import io.lettuce.core.api.sync.RedisCommands;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import io.vertx.core.buffer.Buffer;
import io.vertx.core.buffer.impl.BufferImpl;
import io.vertx.core.streams.ReadStream;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.rest.vertx.BufferReadStream;
import net.bluemind.keydb.common.ClientProvider;
import net.bluemind.lib.vertx.utils.StreamToChunk;
import net.bluemind.memory.pool.api.ChunkFiler;
import net.bluemind.memory.pool.api.IWritableChunk;
import net.bluemind.network.topology.Topology;
import net.bluemind.sds.dto.DeleteRequest;
import net.bluemind.sds.dto.GetRequest;
import net.bluemind.sds.dto.PutRequest;
import net.bluemind.sds.dto.SdsResponse;
import net.bluemind.sds.store.ISdsSyncStore;
import net.bluemind.size.helper.MaxMessageSize;
import net.bluemind.system.sysconf.helper.SysConfHelper;
import net.bluemind.utils.FileUtils;

public class DocumentDbPartFileStore implements IPartFileStore {

	private static final Logger logger = LoggerFactory.getLogger(DocumentDbPartFileStore.class);

	private static final Supplier<DocumentDbPartFileStore> SUPPLIER = Suppliers.memoize(DocumentDbPartFileStore::new);
	private final Supplier<ISdsSyncStore> store;
	private final RedisCommands<String, String> commands;
	private final RedisAsyncCommands<String, String> asyncCommands;
	private static final String PARTS_SET_PREFIX = "partfile-";
	private static final String OBJECT_SUFFIX = ".part";

	private DocumentDbPartFileStore() {
		store = Suppliers.memoize(() -> Topology.getIfAvailable().flatMap(
				topo -> new SdsPartsStoreLoader().forSysconf(SysConfHelper.fromSharedMap(), topo.any("bm/core").uid))//
				.orElseThrow(() -> new ServerFault("Not able to instanciate Sds store.")));

		RedisClient redisClient = ClientProvider.newClient();
		StatefulRedisConnection<String, String> connection = redisClient.connect();
		commands = connection.sync();
		asyncCommands = connection.async();
	}

	public static DocumentDbPartFileStore get() {
		return SUPPLIER.get();
	}

	@Override
	public String save(String sid, ReadStream<Buffer> stream) {
		return save(sid, () -> StreamToChunk.filerAddress(stream, MaxMessageSize.get()));
	}

	@Override
	public String save(String sid, InputStream in) {
		return save(sid, () -> {
			try {
				String addr = ChunkFiler.newChunkAddress(MaxMessageSize.get());
				IWritableChunk chunk = ChunkFiler.byAddress(addr);
				try (OutputStream out = chunk.appendStream()) {
					in.transferTo(out);
				}
				return addr;
			} catch (IOException e) {
				throw new ServerFault(e);
			}
		});
	}

	private String save(String sid, Supplier<String> getTemporarySourcePath) {
		Stopwatch chrono = Stopwatch.createStarted();

		String chunkAddress = getTemporarySourcePath.get();
		String address = createPartFileDocumentDbKey();
		logger.debug("[{}] Upload starts...", address);
		if (commands.sadd(partFileKeyDbKey(sid), address) > 0) {
			SdsResponse resp = store.get().upload(PutRequest.of(address, chunkAddress));
			ChunkFiler.byAddress(chunkAddress).release();
			if (!resp.succeeded()) {
				throw new ServerFault(resp.error.toString());
			}
			Duration elapsed = chrono.elapsed();
			if (elapsed.toMillis() > 500) {
				logger.warn("[{}] Upload Part tooks {}", address, elapsed);
			}
		}
		return address;
	}

	@Override
	public ReadStream<Buffer> get(String address) {
		try {
			logger.debug("Get part {}", address);
			Path tempFile = FileUtils.getTempPath();
			store.get().download(GetRequest.of(null, address, tempFile.toString()));

			try (RandomAccessFile raf = new RandomAccessFile(tempFile.toFile(), "r")) {
				MappedByteBuffer mapped = raf.getChannel().map(MapMode.READ_ONLY, 0, raf.length());
				ByteBuf wrapped = Unpooled.wrappedBuffer(mapped);
				wrapped.readerIndex(0);
				Buffer asVxBuf = BufferImpl.buffer(wrapped);
				return new BufferReadStream(asVxBuf);
			} catch (Exception e) {
				throw new ServerFault(e);
			} finally {
				Files.deleteIfExists(tempFile);
			}
		} catch (IOException e) {
			throw new ServerFault(e);
		}
	}

	@Override
	public void copy(Path path, String address) {
		logger.debug("Copy part {} into {}", address, path);
		SdsResponse resp = store.get().download(GetRequest.of(null, address, path.toString()));
		if (!resp.succeeded()) {
			throw new ServerFault("Could not download " + address + " from " + store.get());
		}
	}

	@Override
	public List<File> getAll(String sid) {
		Set<String> address = commands.smembers(partFileKeyDbKey(sid));
		return address.stream().map((String part) -> {
			Path tempPath = FileUtils.getTempPath();
			store.get().download(GetRequest.of(part, tempPath.toString()));
			return tempPath.toFile();
		}).toList();
	}

	@Override
	public void delete(String sid, String address) {
		if (!address.endsWith(OBJECT_SUFFIX)) {
			logger.error("Prevented deletion of object {}: does not end with .part !!!", address);
			return;
		}

		String key = partFileKeyDbKey(sid);
		asyncCommands.srem(key, address).thenAccept(removedCount -> {
			if (removedCount != null && removedCount > 0) {
				logger.info("Reference to {} successfully removed. Triggering object deletion.", address);
				store.get().delete(DeleteRequest.of(address));
			} else {
				logger.warn("Reference to {} was not in set {}. No object deletion triggered.", address, key);
			}
		}).exceptionally(ex -> {
			logger.error("Asynchronous SREM command failed for part {}", address, ex);
			return null;
		});
	}

	@Override
	public CompletableFuture<Void> deleteAll(String sid) {
		logger.debug("Asynchronously deleting all parts for {} (non-atomic sequence)", sid);
		String key = partFileKeyDbKey(sid);
		return asyncCommands.smembers(key).thenAccept(partFileAdds -> {
			if (partFileAdds == null || partFileAdds.isEmpty()) {
				logger.info("Set {} is empty. Deleting the key just in case.", key);
				asyncCommands.del(key);
				return;
			}
			List<String> partKeys = partFileAdds.stream().filter(partKey -> partKey.endsWith(OBJECT_SUFFIX)).toList();
			if (!partKeys.isEmpty()) {
				logger.info("Deleting {} associated objects for set {}.", partKeys.size(), key);
				SdsResponse response = store.get().delete(DeleteRequest.of(partKeys));
				if (!response.succeeded()) {
					logger.error("Failed to delete objects {}: {}", partKeys, response.error);
				}
			}
			asyncCommands.del(key);
		}).exceptionally(ex -> {
			logger.error("Asynchronous SMEMBERS command failed for key {}", key, ex);
			return null;
		}).toCompletableFuture();
	}

	public static String partFileKeyDbKey(String sid) {
		return new StringBuilder(PARTS_SET_PREFIX).append(sid).toString();
	}

	public static String createPartFileDocumentDbKey() {
		return new StringBuilder(42).append(UUID.randomUUID().toString()).append(OBJECT_SUFFIX).toString();
	}
}
