package net.bluemind.backend.mail.partfile;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.RandomAccessFile;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel.MapMode;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Stopwatch;
import com.google.common.base.Suppliers;

import io.lettuce.core.RedisClient;
import io.lettuce.core.api.StatefulRedisConnection;
import io.lettuce.core.api.sync.RedisCommands;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import io.vertx.core.buffer.Buffer;
import io.vertx.core.buffer.impl.BufferImpl;
import io.vertx.core.streams.ReadStream;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.rest.vertx.BufferReadStream;
import net.bluemind.keydb.common.ClientProvider;
import net.bluemind.network.topology.Topology;
import net.bluemind.sds.dto.DeleteRequest;
import net.bluemind.sds.dto.GetRequest;
import net.bluemind.sds.dto.PutRequest;
import net.bluemind.sds.dto.SdsResponse;
import net.bluemind.sds.store.ISdsSyncStore;
import net.bluemind.system.sysconf.helper.SysConfHelper;
import net.bluemind.utils.FileUtils;
import net.bluemind.utils.FileUtils.TempFileTransfertResult;

public class DocumentDbPartFileStore implements IPartFileStore {

	private static final Logger logger = LoggerFactory.getLogger(DocumentDbPartFileStore.class);

	private static final Supplier<DocumentDbPartFileStore> SUPPLIER = Suppliers.memoize(DocumentDbPartFileStore::new);
	private final Supplier<ISdsSyncStore> store;
	private final RedisCommands<String, String> commands;

	private DocumentDbPartFileStore() {
		store = Suppliers.memoize(() -> Topology.getIfAvailable().flatMap(
				topo -> new SdsPartsStoreLoader().forSysconf(SysConfHelper.fromSharedMap(), topo.any("bm/core").uid))//
				.orElseThrow(() -> new ServerFault("Not able to instanciate Sds store.")));

		RedisClient redisClient = ClientProvider.newClient();
		StatefulRedisConnection<String, String> connection = redisClient.connect();
		commands = connection.sync();
	}

	public static DocumentDbPartFileStore get() {
		return SUPPLIER.get();
	}

	@Override
	public String save(String sid, ReadStream<Buffer> stream) {
		return save(sid, () -> {
			return FileUtils.tempFileTransfert(stream);
		});
	}

	@Override
	public String save(String sid, InputStream in) {
		return save(sid, () -> {
			Path tempPath = FileUtils.getTempPath();
			try {
				Files.copy(in, tempPath);
			} catch (IOException e) {
				throw new ServerFault(e);
			}

			return new TempFileTransfertResult(tempPath, () -> {
				try {
					Files.delete(tempPath);
				} catch (IOException e) {
					logger.error("Not able to delete temp part file", e);
				}
			});
		});
	}

	private String save(String sid, Supplier<TempFileTransfertResult> getTemporarySourcePath) {
		TempFileTransfertResult transfertResult = null;
		try {
			Stopwatch chrono = Stopwatch.createStarted();

			transfertResult = getTemporarySourcePath.get();
			Path tempPath = transfertResult.tempFile();
			logger.debug("Create temporary file {}", tempPath);
			String address = createPartFileDocumentDbKey();
			logger.debug("[{}] Upload starts...", address);
			if (commands.sadd(partFileKeyDbKey(sid), address) > 0) {
				SdsResponse resp = store.get().upload(PutRequest.of(address, tempPath.toString()));
				if (!resp.succeeded()) {
					throw new ServerFault(resp.error.toString());
				}
				long elapsed = chrono.elapsed(TimeUnit.MILLISECONDS);
				if (elapsed > 500) {
					logger.warn("[{}] Upload Part tooks {}", address, elapsed);
				}
			}
			return address;
		} finally {
			if (transfertResult != null) {
				transfertResult.cleanUp().run();
			}
		}
	}

	@Override
	public ReadStream<Buffer> get(String address) {
		try {
			logger.debug("Get part {}", address);
			Path tempFile = FileUtils.getTempPath();
			store.get().download(GetRequest.of(null, address, tempFile.toString()));

			try (RandomAccessFile raf = new RandomAccessFile(tempFile.toFile(), "r")) {
				MappedByteBuffer mapped = raf.getChannel().map(MapMode.READ_ONLY, 0, raf.length());
				ByteBuf wrapped = Unpooled.wrappedBuffer(mapped);
				wrapped.readerIndex(0);
				Buffer asVxBuf = BufferImpl.buffer(wrapped);
				return new BufferReadStream(asVxBuf);
			} catch (Exception e) {
				throw new ServerFault(e);
			} finally {
				Files.deleteIfExists(tempFile);
			}
		} catch (IOException e) {
			throw new ServerFault(e);
		}
	}

	@Override
	public void copy(Path path, String address) {
		logger.debug("Copy part {} into {}", address, path);
		SdsResponse resp = store.get().download(GetRequest.of(null, address, path.toString()));
		if (!resp.succeeded()) {
			throw new ServerFault("Could not download " + address + " from " + store.get());
		}
	}

	@Override
	public List<File> getAll(String sid) {
		Set<String> address = commands.smembers(partFileKeyDbKey(sid));
		return address.stream().map((String part) -> {
			Path tempPath = FileUtils.getTempPath();
			store.get().download(GetRequest.of(part, tempPath.toString()));
			return tempPath.toFile();
		}).toList();
	}

	@Override
	public void delete(String sid, String address) {
		logger.debug("Delete part {}", address);
		store.get().delete(DeleteRequest.of(address));
		commands.srem(partFileKeyDbKey(sid), address);
	}

	@Override
	public void deleteAll(String sid) {
		logger.debug("Delete all part for {}", sid);
		Set<String> partFileAdds = commands.smembers(partFileKeyDbKey(sid));
		for (String partFileAdd : partFileAdds) {
			SdsResponse response = store.get().delete(DeleteRequest.of(partFileAdd));
			if (!response.succeeded()) {
				logger.error("Not able to delete {} part file", partFileAdd);
			}
		}
		commands.del(partFileKeyDbKey(sid));
	}

	public static String partFileKeyDbKey(String sid) {
		return new StringBuilder("partfile-").append(sid).toString();
	}

	public static String createPartFileDocumentDbKey() {
		return new StringBuilder(42).append(UUID.randomUUID().toString()).append(".part").toString();
	}
}
