/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2022
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.backend.mail.parsing;

import java.net.URLDecoder;
import java.nio.charset.Charset;
import java.nio.charset.IllegalCharsetNameException;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.james.mime4j.dom.Entity;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.mime4j.common.AddressableEntity;

public class Rfc2231Decoder {
	private static final Logger logger = LoggerFactory.getLogger(Rfc2231Decoder.class);
	private static final Pattern RFC2231_PATTERN = Pattern.compile("^([^']*)'([^']*)'(.*)$");

	public static String decodeFileName(Entity entity) {
		return Optional.ofNullable(AddressableEntity.getFileName(entity)).map(Rfc2231Decoder::decodeRfc2231Token)
				.orElse(null);
	}

	public static String decodeRfc2231Token(String token) {
		Matcher matcher = RFC2231_PATTERN.matcher(token);
		if (matcher.matches()) {
			try {
				String charsetName = matcher.group(1);
				if (charsetName != null && !charsetName.isBlank() && !Charset.isSupported(charsetName)) {
					return token;
				}
				String content = matcher.group(3);
				return URLDecoder.decode(content,
						charsetName != null && !charsetName.isBlank() ? charsetName : "UTF-8");
			} catch (IllegalCharsetNameException ice) {
				return token;
			} catch (Exception e) {
				logger.warn("{} has been identified as an RFC2231 encoded token, but decoding has failed", token, e);
				return token;
			}
		} else {
			return token;
		}

	}
}
