/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of Blue Mind. Blue Mind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License)
  * or the CeCILL as published by CeCILL.info (version 2 of the License).
  *
  * There are special exceptions to the terms and conditions of the
  * licenses as they are applied to this program. See LICENSE.txt in
  * the directory of this program distribution.
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.authentication.service.logout;

import java.util.Optional;

import net.bluemind.core.sessions.Sessions;
import net.bluemind.keycloak.utils.sessions.JwtManager;
import net.bluemind.keydb.sessiondata.SessionData;
import net.bluemind.keydb.sessiondata.SessionDataStore;

public class LogoutAction {
	private final String sessionId;
	private final Optional<JwtManager> jwtManager;

	private LogoutAction(String sessionId) {
		this.sessionId = sessionId;
		this.jwtManager = Optional.empty();
	}

	public LogoutAction(String sessionId, Optional<JwtManager> jwtManager) {
		this.sessionId = sessionId;
		this.jwtManager = jwtManager;
	}

	public static LogoutAction coreOnly(String sessionId) {
		return new LogoutAction(sessionId);
	}

	public static LogoutAction full(String sessionId) {
		SessionData sessionData = SessionDataStore.get().getIfPresent(sessionId);
		if (sessionData == null || !sessionData.internalAuth) {
			return coreOnly(sessionId);
		}

		return new LogoutAction(sessionId, JwtManager.fromJson(sessionData.realm, sessionData.jwtToken));
	}

	/**
	 * Try KC logout if possible - prefer backchannel logout, fallback to core
	 * logout
	 */
	public void logout() {
		jwtManager.ifPresentOrElse(JwtManager::logoutUserSession, this::coreLogout);
	}

	/**
	 * Do core only logout
	 */
	private void coreLogout() {
		Sessions.get().invalidate(sessionId);
	}
}
