/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.authentication.service;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.function.Predicate;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Splitter;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableSet;

import net.bluemind.authentication.api.AuthUser;
import net.bluemind.authentication.api.LoginResponse;
import net.bluemind.authentication.api.LoginResponse.Status;
import net.bluemind.authentication.api.ValidationKind;
import net.bluemind.authentication.api.incore.IInCoreAuthentication;
import net.bluemind.authentication.provider.IAuthProvider;
import net.bluemind.authentication.provider.IAuthProvider.AuthResult;
import net.bluemind.authentication.provider.ILoginSessionValidator;
import net.bluemind.authentication.provider.ILoginValidationListener;
import net.bluemind.authentication.service.internal.AuthContext;
import net.bluemind.authentication.service.internal.UserCache;
import net.bluemind.authentication.service.logout.LogoutAction;
import net.bluemind.core.api.Stream;
import net.bluemind.core.api.fault.ErrorCode;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.container.service.internal.RBACManager;
import net.bluemind.core.container.service.internal.SecurityContextAuditLogService;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.rest.IServiceProvider;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.core.rest.base.GenericStream;
import net.bluemind.core.sessions.Sessions;
import net.bluemind.directory.api.IOrgUnits;
import net.bluemind.directory.api.OrgUnitPath;
import net.bluemind.domain.api.Domain;
import net.bluemind.domain.api.IDomainUids;
import net.bluemind.domain.api.IDomains;
import net.bluemind.keycloak.utils.sessions.BackchannelLogoutToken;
import net.bluemind.keydb.sessiondata.SessionData;
import net.bluemind.keydb.sessiondata.SessionDataStore;
import net.bluemind.role.api.BasicRoles;
import net.bluemind.role.api.DefaultRoles;
import net.bluemind.role.service.IInternalRoles;
import net.bluemind.system.api.SystemState;
import net.bluemind.system.state.StateContext;
import net.bluemind.user.api.IUser;
import net.bluemind.user.api.IUserSettings;
import net.bluemind.user.api.User;
import net.bluemind.user.api.UserProperties;
import net.bluemind.user.service.IInCoreUser;

public class Authentication implements IInCoreAuthentication {
	private static final Logger logger = LoggerFactory.getLogger(Authentication.class);

	private static final Splitter atSplitter = Splitter.on('@').trimResults().omitEmptyStrings();
	private static final String AUDIT_AUTH_ERROR = "Error with authentication auditlog: {}";

	private final SecurityContext securityContext;
	private final List<IAuthProvider> authProviders;
	private final List<ILoginValidationListener> loginListeners;
	private final List<ILoginSessionValidator> sessionValidators;
	private final IDomains domainService;
	private final Supplier<Optional<SecurityContextAuditLogService>> auditLogServiceSupplier;

	private record ResultAndProvider(AuthResult authResult, IAuthProvider provider) {
	}

	private record SessionInfos(String locale, String origin) {
	}

	private BmContext context;

	public Authentication(BmContext context, List<IAuthProvider> authProviders,
			List<ILoginValidationListener> loginListeners, List<ILoginSessionValidator> sessionValidators,
			SecurityContextAuditLogService auditLogService) throws ServerFault {
		this.context = context;
		this.securityContext = context.getSecurityContext();
		this.authProviders = authProviders;
		this.loginListeners = loginListeners;
		this.sessionValidators = sessionValidators;
		this.auditLogServiceSupplier = () -> {
			if (StateContext.getState().equals(SystemState.CORE_STATE_RUNNING)) {
				return Optional.of(auditLogService);
			}
			return Optional.empty();
		};
		this.domainService = context.su().provider().instance(IDomains.class);
	}

	@Override
	public LoginResponse login(String login, String password, String origin) throws ServerFault {
		return loginWithParams(login, password, origin, false);
	}

	@Override
	public LoginResponse loginWithParams(String login, String password, String origin, Boolean interactive)
			throws ServerFault {
		if (!verifyNonEmptyCredentials(login, password, origin)) {
			LoginResponse resp = new LoginResponse();
			resp.status = Status.Bad;
			return resp;
		}

		logger.debug("try login with l: '{}', o: '{}'", login, origin);

		SystemState systemState = StateContext.getState();
		if (systemState != SystemState.CORE_STATE_RUNNING && !context.getSecurityContext().isAdmin()
				&& !"admin0@global.virt".equals(login)) {
			LoginResponse maintenanceResponse = new LoginResponse();
			maintenanceResponse.status = Status.Bad;
			logger.warn("Authentication denied for user {} while system is in maintenance mode", login);
			maintenanceResponse.message = "Authentication denied while system is in maintenance mode";
			return maintenanceResponse;
		}

		Optional<AuthContext> authContext = buildAuthContext(login, password);

		// Is user archived ?
		if (authContext.map(ac -> ac.user).filter(u -> u.value.archived).isPresent()) {
			return authContextNotFoundResponse(origin, login);
		}

		SecurityContext sc = Sessions.get().getIfPresent(password);
		AuthResult authResult = authContext.map(ac -> checkToken(sc, ac)).orElse(AuthResult.UNKNOWN);
		if (authResult != AuthResult.YES) {
			// checkProviders are able to create user on the fly
			// If AuthContext is null, try to build a fake AuthContext from login
			// If fake AuthContext is null too, do not try checkProviders
			AuthContext providerAuthContext = authContext
					.orElseGet(() -> AuthContext.buildFakeAuthContext(domainService, login, password));
			if (providerAuthContext != null) {
				try {
					authResult = checkProviders(providerAuthContext, acceptUserPassword(providerAuthContext.getUser()),
							origin);
				} catch (Exception e) {
					logger.error("Unable to check auth provider for {}", login, e);
					return authContextNotFoundResponse(origin, login);
				}
			}
		}

		// user created on the fly ?
		// re-try to build AuthContext if null and AuthResult is ok
		if ((authResult == AuthResult.YES || authResult == AuthResult.EXPIRED) && !authContext.isPresent()) {
			authContext = buildAuthContext(login, password);
		}

		AuthResult finalAuthResult = authResult;
		return authContext
				.map(ac -> getLoginResponse(sc, origin, Boolean.TRUE.equals(interactive), ac, login, finalAuthResult))
				.orElseGet(() -> authContextNotFoundResponse(origin, login));
	}

	private AuthResult checkToken(SecurityContext sc, AuthContext authContext) {
		if (sc == null) {
			return AuthResult.UNKNOWN;
		}

		if (authContext.user == null || !authContext.user.uid.equals(sc.getSubject())) {
			logger.debug("login with token by {} but user doesnt match session", authContext.user);
			return AuthResult.UNKNOWN;
		}

		logger.debug("login with token by {}", authContext.user);
		return AuthResult.YES;
	}

	private LoginResponse getLoginResponse(SecurityContext sc, String origin, boolean interactive,
			AuthContext authContext, String login, AuthResult authResult) {
		LoginResponse loginResponse = null;
		switch (authResult) {
		case YES:
			loginResponse = getLoginResponse(sc, origin, interactive, authContext, login, Status.Ok);
			break;
		case EXPIRED:
			loginResponse = getLoginResponse(sc, origin, interactive, authContext, login, Status.Expired);
			break;
		default:
			logger.error("Result auth is {} for login: {} origin: {} remoteIps: {}", authResult, login, origin,
					securityContext.getRemoteAddresses());
			String loginResponseMessage = "Result auth is " + authResult.name() + " for login: " + login;
			loginResponse = new LoginResponse();
			loginResponse.status = Status.Bad;
			loginResponse.message = loginResponseMessage;

			auditLogServiceSupplier.get().ifPresent(auditLogService -> {
				try {
					auditLogService.logCreate(sc, authContext.domain.uid, loginResponseMessage);
				} catch (Exception e) {
					logger.error(AUDIT_AUTH_ERROR, e.getMessage());
				}
			});
		}

		if (loginResponse.status != Status.Bad && interactive) {
			SessionDataStore.get().put(new SessionData(loginResponse));
		}

		return loginResponse;
	}

	private LoginResponse authContextNotFoundResponse(String origin, String login) {
		if (logger.isInfoEnabled()) {
			logger.error("authContext.user not constructed for login: {} origin: {} remoteIps: {}", login, origin,
					String.join(",", securityContext.getRemoteAddresses()));
		}

		LoginResponse resp = new LoginResponse();
		resp.status = Status.Bad;
		return resp;
	}

	private LoginResponse getLoginResponse(SecurityContext sec, String origin, boolean interactive,
			AuthContext authContext, String login, Status status) {
		LoginResponse resp = new LoginResponse();
		resp.status = status;
		resp.latd = authContext.user.value.login + "@" + authContext.domain.uid;

		IServiceProvider sp = context.su().provider();
		IUserSettings userSettingsService = sp.instance(IUserSettings.class, authContext.domain.uid);
		Map<String, String> settings = userSettingsService.get(authContext.user.uid);

		if (sec == null) {
			logger.info("[name={};origin={};status={}] authenticated", login, origin, status);
			resp.authKey = UUID.randomUUID().toString();
			IUser userService = sp.instance(IUser.class, authContext.domain.uid);
			sec = buildSecurityContext(resp.authKey, authContext.user, authContext.domain.uid, status == Status.Expired,
					interactive, new SessionInfos(userService.getLocale(authContext.user.uid), origin));

			for (ILoginSessionValidator v : sessionValidators) {
				try {
					sec = v.validateAndModifySession(sec);
				} catch (ServerFault e) {
					resp.status = Status.Bad;
					resp.message = e.getMessage();
					return resp;
				}
			}

			if (logger.isDebugEnabled()) {
				logger.debug("[{}] authentified with token : {}", login, sec);
			}

			Sessions.get().put(resp.authKey, sec);
		} else {

			logger.debug("[name={};origin={};from={}] authenticated session token", login, origin,
					securityContext.getRemoteAddresses());
			resp.authKey = sec.getSessionId();
		}
		final SecurityContext finalContext = sec;

		auditLogServiceSupplier.get().ifPresent(auditLogService -> {
			try {
				auditLogService.logCreate(finalContext, authContext.domain.uid, resp.message);
			} catch (Exception e) {
				logger.error(AUDIT_AUTH_ERROR, e.getMessage());
			}
		});

		resp.authUser = AuthUser.create(sec.getContainerUid(), sec.getSubject(), authContext.user.displayName,
				authContext.user.value, new HashSet<>(sec.getRoles()), sec.getRolesByOrgUnits(), settings);
		return resp;
	}

	private boolean verifyNonEmptyCredentials(String login, String password, String origin) {
		if (Strings.isNullOrEmpty(login)) {
			if (logger.isErrorEnabled()) {
				logger.error("Empty login forbidden from {}, remote IPs {}", origin,
						String.join(",", securityContext.getRemoteAddresses()));
			}
			return false;
		}

		if (Strings.isNullOrEmpty(password)) {
			if (logger.isErrorEnabled()) {
				logger.error("Empty password forbidden for login: {} from {}, remote IPs {}", login, origin,
						String.join(",", securityContext.getRemoteAddresses()));
			}
			return false;
		}

		return true;
	}

	private Optional<AuthContext> buildAuthContext(String login, String password) throws ServerFault {
		Iterator<String> splitted = atSplitter.split(login).iterator();
		String localPart = splitted.next();
		String domainPart = splitted.hasNext() ? splitted.next() : IDomainUids.GLOBAL_VIRT;

		ItemValue<Domain> theDomain = domainService.findByNameOrAliases(domainPart);
		if (theDomain == null) {
			logger.error("Domain {} not found.", domainPart);
			return Optional.empty();
		}

		ItemValue<User> user = UserCache.getCache().get(login,
				l -> getUserByLoginOrEmail(login, localPart, domainPart, theDomain.uid));
		if (user == null) {
			return Optional.empty();
		}

		return Optional.of(new AuthContext(context.getSecurityContext(), theDomain, user, password));
	}

	private ItemValue<User> getUserByLoginOrEmail(String login, String localPart, String domainPart, String domainUid) {
		IUser userService = context.su().provider().instance(IUser.class, domainUid);

		ItemValue<User> internalUser = null;

		if (domainPart.equals(IDomainUids.GLOBAL_VIRT) || domainPart.equals(domainUid)) {
			internalUser = userService.byLogin(localPart);
		}

		if (internalUser == null) {
			internalUser = userService.byEmail(login);
		}

		if (internalUser == null) {
			logger.warn("no user found for login {}", login);
		} else {
			if (logger.isDebugEnabled()) {
				logger.debug("found user {}, domain {} for login {}", internalUser.value.login, domainUid, login);
			}
		}

		return internalUser;
	}

	private AuthResult checkProviders(AuthContext authContext, boolean acceptPassword, String origin) {
		if (logger.isDebugEnabled()) {
			logger.debug("[{}@{}] Auth attempt from {}", authContext.localPart,
					authContext.domain != null ? authContext.domain.value.name : "null domain", origin);
		}

		if (authContext.domain == null) {
			logger.info("[{}] authenticate failed: domain is null", authContext.getRealUserLogin());
			return AuthResult.NO;
		}

		ResultAndProvider resultAndProvider = authProviders.stream()
				.filter(provider -> acceptPassword || !provider.checkPassword()).map(provider -> {
					AuthResult authResult = provider.check(context, authContext);
					if (logger.isDebugEnabled()) {
						logger.debug("[{}@{}] {} result: {}", authContext.getRealUserLogin(),
								authContext.domain.value.name, provider, authResult);
					}

					return new ResultAndProvider(authResult, provider);
				}).filter(rp -> rp.authResult != AuthResult.UNKNOWN).findFirst()
				.orElseGet(() -> new ResultAndProvider(AuthResult.UNKNOWN, null));

		if (resultAndProvider.authResult == AuthResult.YES || resultAndProvider.authResult == AuthResult.EXPIRED) {
			loginListeners.forEach(vl -> vl.onValidLogin(resultAndProvider.provider, authContext.user != null,
					authContext.getRealUserLogin(), authContext.domain.uid, authContext.userPassword));
		} else if (resultAndProvider.authResult == AuthResult.NO) {
			loginListeners.forEach(vl -> vl.onFailedLogin(resultAndProvider.provider, authContext.user != null,
					authContext.getRealUserLogin(), authContext.domain.uid, authContext.userPassword));
		}

		if (logger.isDebugEnabled()) {
			logger.debug("[{}@{}] authenticate: {}", authContext.getRealUserLogin(), authContext.domain.value.name,
					resultAndProvider.authResult);
		}

		return resultAndProvider.authResult;
	}

	@Override
	public void logout() throws ServerFault {
		if (securityContext.getSessionId() == null) {
			logger.debug("try to logout without session");
			return;
		}

		if (logger.isDebugEnabled()) {
			logger.debug("logout user {} session {}", securityContext.getSubject(), securityContext.getSessionId());
		}

		LogoutAction.full(securityContext.getSessionId()).logout();
	}

	@Override
	public void backchannelLogout(Stream contentStream) {
		String content = GenericStream.streamToString(contentStream);

		if (!content.startsWith("logout_token=")) {
			throw new ServerFault("No logout token found", ErrorCode.BAD_REQUEST);
		}

		BackchannelLogoutToken logoutToken = BackchannelLogoutToken
				.fromString(content.substring("logout_token=".length()));
		String ssid = null;
		if (logoutToken == null || (ssid = logoutToken.getSid()) == null) {
			throw new ServerFault("Invalid logout token", ErrorCode.BAD_REQUEST);
		}

		SessionData sessionData = SessionDataStore.get().getFromSessionState(ssid);
		if (sessionData == null) {
			if (logger.isWarnEnabled()) {
				logger.warn("Backchannel logout: session not found for session state ID '{}'", ssid);
			}

			return;
		}

		LogoutAction.coreOnly(sessionData.authKey).logout();
	}

	@Override
	public LoginResponse su(String login) throws ServerFault {
		return suWithParams(login, false);
	}

	@Override
	public LoginResponse suWithParams(String login, Boolean inter) throws ServerFault {
		boolean interactive = Boolean.TRUE.equals(inter);
		String performer = securityContext.getSubject();
		if (interactive && !securityContext.isDomainGlobal()) {
			AuthUser currentUser = getCurrentUser();
			performer = String.format("%s (%s)", securityContext.getSubject(), currentUser.displayName);
		}
		List<String> oips = new ArrayList<>();
		for (String ipAddress : securityContext.getRemoteAddresses()) {
			if (!ipAddress.startsWith("127.")) {
				oips.add(ipAddress);
			}
		}
		if (logger.isInfoEnabled()) {
			logger.info("[name={};origin={};oip={}] sudo as by {}", login, securityContext.getOrigin(),
					String.join(",", oips), performer);
		}

		Iterator<String> splitted = atSplitter.split(login).iterator();
		String localPart = splitted.next();
		String domainPart = splitted.next();

		ItemValue<Domain> domain = domainService.findByNameOrAliases(domainPart);
		if (domain == null) {
			logger.error("Cannot find domain alias {}", domainPart);
			LoginResponse resp = new LoginResponse();
			resp.status = Status.Bad;
			return resp;
		}

		ItemValue<User> user;
		try {
			user = findOrGetUser(domain, localPart);
		} catch (ServerFault sf) {
			LoginResponse resp = new LoginResponse();
			resp.status = Status.Bad;
			return resp;
		}

		if (user == null) {
			logger.error("Cannot find user with login {} in {}", localPart, domainPart);
			LoginResponse resp = new LoginResponse();
			resp.status = Status.Bad;
			return resp;
		} else if (user.value.archived && !(securityContext.getOrigin().equals("mapi-admin-link")
				|| securityContext.getOrigin().equals("internal-system"))) {
			logger.error("user with login {} in {} is archived, su refused", localPart, domainPart);
			LoginResponse resp = new LoginResponse();
			resp.status = Status.Bad;
			return resp;
		}

		return doSu(interactive, domain, user);
	}

	private ItemValue<User> findOrGetUser(ItemValue<Domain> domain, String localPart) {
		IUser userService = ServerSideServiceProvider.getProvider(securityContext).instance(IUser.class, domain.uid);

		ItemValue<User> user = userService.byLogin(localPart);
		if (user == null) {
			user = loginListeners.stream().map(vl -> vl.onSu(domain, localPart)).filter(Objects::nonNull).findAny()
					.orElse(null);
		}

		return user;
	}

	private LoginResponse doSu(boolean interactive, ItemValue<Domain> domain, ItemValue<User> user) {
		new RBACManager(context).forDomain(domain.uid).forEntry(user.uid).check(BasicRoles.ROLE_SUDO);

		LoginResponse loginResponse = new LoginResponse();
		loginResponse.latd = user.value.login + "@" + domain.uid;
		loginResponse.status = Status.Ok;
		loginResponse.authKey = UUID.randomUUID().toString();

		ServerSideServiceProvider sp = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM);
		IUserSettings userSettingsService = sp.instance(IUserSettings.class, domain.uid);
		IUser userService = sp.instance(IUser.class, domain.uid);
		Map<String, String> settings = userSettingsService.get(user.uid);

		SecurityContext builtContext = buildSecurityContext(loginResponse.authKey, user, domain.uid, false, interactive,
				new SessionInfos(userService.getLocale(user.uid), securityContext.getOrigin()));

		auditLogServiceSupplier.get().ifPresent(auditLogService -> {
			try {
				auditLogService.logCreate(builtContext, domain.uid, loginResponse.message);
			} catch (Exception e) {
				logger.error(AUDIT_AUTH_ERROR, e.getMessage());
			}
		});

		loginResponse.authUser = AuthUser.create(builtContext.getContainerUid(), builtContext.getSubject(),
				user.displayName, user.value, new HashSet<>(builtContext.getRoles()), builtContext.getRolesByOrgUnits(),
				settings);
		Sessions.get().put(loginResponse.authKey, builtContext);

		if (loginResponse.status != Status.Bad && interactive) {
			SessionDataStore.get().put(new SessionData(loginResponse));
		}

		return loginResponse;
	}

	@Override
	public AuthUser getCurrentUser() throws ServerFault {
		RBACManager.forContext(context).checkNotAnoynmous();

		IUser userService = context.provider().instance(IUser.class, securityContext.getContainerUid());

		ItemValue<User> userItem = userService.getComplete(securityContext.getSubject());

		if (userItem == null) {
			logger.error("userItem of the current user is null");
			return null;
		}
		Map<String, String> settings = context.su().provider()
				.instance(IUserSettings.class, securityContext.getContainerUid()).get(userItem.uid);

		return AuthUser.create(securityContext.getContainerUid(), securityContext.getSubject(), userItem.displayName,
				userItem.value, new HashSet<>(securityContext.getRoles()), securityContext.getRolesByOrgUnits(),
				settings);
	}

	@Override
	public void ping() throws ServerFault {
		// This method is empty as {@link Sessions#sessionContext(String)} is called
		// from the rest layer.
	}

	private SecurityContext buildSecurityContext(String authKey, ItemValue<User> user, String domainUid,
			boolean expiredPassword, boolean interactive, SessionInfos sessionInfos) throws ServerFault {
		IServiceProvider sp = context.su().provider();
		List<String> groups = sp.instance(IUser.class, domainUid).memberOfGroups(user.uid);

		Map<String, Set<String>> rolesByOUs = Collections.emptyMap();
		if ((!expiredPassword || IDomainUids.GLOBAL_VIRT.equals(domainUid)) && user.value.fullAccount()) {
			IOrgUnits orgUnits = sp.instance(IOrgUnits.class, domainUid);
			List<OrgUnitPath> ous = orgUnits.listByAdministrator(user.uid, groups);
			rolesByOUs = ous.stream() //
					.collect(Collectors.toMap( //
							ouPath -> ouPath.uid, // key: orgUnit.uid
							ouPath -> orgUnits.getAdministratorRoles(ouPath.uid, user.uid, groups))); // roles
		}

		return new SecurityContext(authKey, user.uid, user.value.login + "@" + domainUid, groups,
				new ArrayList<>(getRoles(domainUid, user.uid, groups, expiredPassword)), rolesByOUs, domainUid,
				sessionInfos.locale, sessionInfos.origin, interactive);
	}

	private Set<String> getRoles(String domainUid, String userUid, List<String> groups, boolean expiredPassword)
			throws ServerFault {
		IServiceProvider sp = context.su().provider();
		if (IDomainUids.GLOBAL_VIRT.equals(domainUid)) {
			return sp.instance(IInternalRoles.class).resolve(ImmutableSet.<String>builder()
					.add(SecurityContext.ROLE_SYSTEM).add(BasicRoles.ROLE_SELF_CHANGE_PASSWORD).build());
		} else {
			if (expiredPassword) {
				return DefaultRoles.USER_PASSWORD_EXPIRED;
			}

			return sp.instance(IInCoreUser.class, domainUid).directResolvedRoles(userUid, groups).stream()
					.filter(role -> RoleValidation.validate(domainUid, role)).collect(Collectors.toSet());
		}
	}

	@Override
	public SecurityContext buildContext(String sid, String origin, String domainUid, String userUid)
			throws ServerFault {
		ServerSideServiceProvider sp = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM);

		IUser userService = sp.instance(IUser.class, domainUid);
		ItemValue<User> user = userService.getComplete(userUid);

		return buildSecurityContext(sid, user, domainUid, false, false,
				new SessionInfos(userService.getLocale(userUid), origin));
	}

	@Override
	public ValidationKind validate(String login, String password, String origin) throws ServerFault {
		return validatePassword(login, password, origin, this::acceptUserPassword);
	}

	@Override
	public ValidationKind validateCredentials(String login, String origin, String password) throws ServerFault {
		return validatePassword(login, password, origin, user -> true);
	}

	private boolean acceptUserPassword(ItemValue<User> user) {
		return Optional.ofNullable(user)
				.map(u -> !Boolean.valueOf(u.value.properties.get(UserProperties.NO_PASSWD_AUTH.name()))).orElse(true);
	}

	private ValidationKind validatePassword(String login, String password, String origin,
			Predicate<ItemValue<User>> acceptUserPassword) {
		if (!verifyNonEmptyCredentials(login, password, origin)) {
			return ValidationKind.NONE;
		}

		AuthContext authContext = buildAuthContext(login, password).orElse(null);
		if (authContext == null || authContext.user == null) {
			logger.error("validate failed for login: {} origin: {} remoteIps: {}", login, origin,
					securityContext.getRemoteAddresses());
			return ValidationKind.NONE;
		}

		// check session
		SecurityContext cachedContext = Sessions.get().getIfPresent(password);
		if (cachedContext != null && cachedContext.getSessionId().equals(password)
				&& cachedContext.getSubject().equals(authContext.user.uid)) {
			return ValidationKind.TOKEN;
		}

		AuthResult authResult = checkProviders(authContext, acceptUserPassword.test(authContext.user), origin);
		if (authResult == AuthResult.YES) {
			return ValidationKind.PASSWORD;
		} else if (authResult == AuthResult.EXPIRED) {
			return ValidationKind.PASSWORDEXPIRED;
		} else if (authResult == AuthResult.ARCHIVED) {
			return ValidationKind.ARCHIVED;
		}

		logger.error("validate password or token failed for login: {} result: {} origin: {} remoteIps: {}", login,
				authResult, origin, securityContext.getRemoteAddresses());
		auditLogServiceSupplier.get().ifPresent(auditLogService -> {
			try {
				auditLogService.logCreate((authContext != null) ? authContext.securityContext : null,
						authContext.domain.uid, "authorization result: " + authResult.name());
			} catch (Exception e) {
				logger.error(AUDIT_AUTH_ERROR, e.getMessage());
			}
		});

		return ValidationKind.NONE;
	}
}
