/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.authentication.service;

import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.UUID;

import com.google.common.hash.Hashing;

import net.bluemind.authentication.api.APIKey;
import net.bluemind.authentication.api.IInternalAPIKeys;
import net.bluemind.authentication.repository.IAPIKeyStore;
import net.bluemind.core.api.Regex;
import net.bluemind.core.api.fault.ErrorCode;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.DataLocation;
import net.bluemind.core.container.model.Item;
import net.bluemind.core.container.model.ItemFlag;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.tx.wrapper.TxEnabler;
import net.bluemind.repository.provider.RepositoryProvider;
import net.bluemind.system.api.ICredentials;
import net.bluemind.tx.outbox.api.ITxOutbox;

public class APIKeysService implements IInternalAPIKeys {
	private static final String INVALID_SECURITY_CONTEXT = "Invalid securityContext";
	private final IAPIKeyStore store;
	private final SecurityContext context;
	private ITxOutbox outbox;

	public APIKeysService(BmContext bmCtx) {
		store = RepositoryProvider.instance(IAPIKeyStore.class, bmCtx);
		this.context = bmCtx.getSecurityContext();
		this.outbox = bmCtx.su().provider().instance(ITxOutbox.class, "global.virt", "system", "apikeys", "apikeys",
				DataLocation.directory().serverUid());
	}

	private Item item(String uid) {
		return Item.create(uid, Hashing.sipHash24().hashString(uid, StandardCharsets.US_ASCII).asLong(), ItemFlag.Seen);
	}

	@Override
	public APIKey create(String displayName) throws ServerFault {
		if (context.isAnonymous()) {
			throw new ServerFault(INVALID_SECURITY_CONTEXT, ErrorCode.PERMISSION_DENIED);
		}

		APIKey apikey = createApiKey(displayName, UUID.randomUUID().toString());
		create(apikey);

		return store.getBySid(apikey.sid);
	}

	@Override
	public void delete(String uid) throws ServerFault {
		if (context.isAnonymous()) {
			throw new ServerFault(INVALID_SECURITY_CONTEXT, ErrorCode.PERMISSION_DENIED);
		}

		TxEnabler.atomically(() -> {
			store.delete(uid);
			outbox.forKafka(ItemValue.create(item(uid), new APIKey()), null, true);
		});
	}

	@Override
	public List<APIKey> list() throws ServerFault {
		if (context.isAnonymous()) {
			throw new ServerFault(INVALID_SECURITY_CONTEXT, ErrorCode.PERMISSION_DENIED);
		}

		return store.list().stream().map(apiKey -> {
			apiKey.sid = ICredentials.OBFUSCATED_CREDENTIAL_VALUE;
			return apiKey;
		}).toList();
	}

	@Override
	public String put(String displayName, String sid) throws ServerFault {
		if (context.isAnonymous()) {
			throw new ServerFault(INVALID_SECURITY_CONTEXT, ErrorCode.PERMISSION_DENIED);
		}

		if (!Regex.UUID.validate(sid)) {
			throw new ServerFault("API key SID must be an UUID");
		}

		APIKey apikey = createApiKey(displayName, sid);
		create(apikey);

		return apikey.uid;
	}

	@Override
	public APIKey get(String sid) throws ServerFault {
		return store.getBySid(sid);
	}

	private void create(APIKey apikey) {
		TxEnabler.atomically(() -> {
			store.create(apikey);
			outbox.forKafka(ItemValue.create(item(apikey.uid), apikey), null, false);
		});
	}

	private String validateDisplayName(String displayName) {
		if (displayName == null || displayName.isBlank()) {
			throw new ServerFault("API key display name cannot be empty");
		}

		return displayName.trim();
	}

	private APIKey createApiKey(String displayName, String sid) {
		APIKey apikey = new APIKey();
		apikey.uid = UUID.randomUUID().toString();
		apikey.sid = sid;
		apikey.displayName = validateDisplayName(displayName);
		apikey.subject = context.getSubject();
		apikey.domainUid = context.getContainerUid();

		return apikey;
	}
}
