/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of Blue Mind. Blue Mind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License)
  * or the CeCILL as published by CeCILL.info (version 2 of the License).
  *
  * There are special exceptions to the terms and conditions of the
  * licenses as they are applied to this program. See LICENSE.txt in
  * the directory of this program distribution.
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.authentication.service.internal;

import java.util.Iterator;

import com.google.common.base.Splitter;

import net.bluemind.authentication.provider.IAuthProvider.IAuthContext;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.domain.api.Domain;
import net.bluemind.domain.api.IDomains;
import net.bluemind.user.api.User;

public class AuthContext implements IAuthContext {
	private static final Splitter atSplitter = Splitter.on('@').trimResults().omitEmptyStrings();

	public final SecurityContext securityContext;
	public final ItemValue<Domain> domain;
	public final ItemValue<User> user;
	public final String localPart;
	public final String userPassword;

	public AuthContext(SecurityContext securityContext, ItemValue<Domain> domain, ItemValue<User> user,
			String userPassword) {
		this.securityContext = securityContext;
		this.domain = domain;
		this.user = user;
		this.localPart = null;
		this.userPassword = userPassword;
	}

	private AuthContext(ItemValue<Domain> domain, String localPart, String password) {
		this.securityContext = null;
		this.domain = domain;
		this.user = null;
		this.localPart = localPart;
		this.userPassword = password;
	}

	public static AuthContext buildFakeAuthContext(IDomains domainService, String login, String password) {
		Iterator<String> splitted = atSplitter.split(login).iterator();
		String localPart = splitted.next();

		if (!splitted.hasNext()) {
			return null;
		}

		return new AuthContext(domainService.findByNameOrAliases(splitted.next()), localPart, password);
	}

	@Override
	public SecurityContext getSecurityContext() {
		return securityContext;
	}

	@Override
	public ItemValue<Domain> getDomain() {
		return domain;
	}

	@Override
	public ItemValue<User> getUser() {
		return user;
	}

	@Override
	public String getRealUserLogin() {
		return user != null ? user.value.login : localPart;
	}

	@Override
	public String getUserPassword() {
		return userPassword;
	}
}
