/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.authentication.mgmt.service;

import java.util.List;
import java.util.Optional;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.authentication.mgmt.api.ISessionsMgmt;
import net.bluemind.authentication.mgmt.api.SessionEntry;
import net.bluemind.authentication.mgmt.api.SessionUpdate;
import net.bluemind.core.api.auth.AuthDomainProperties;
import net.bluemind.core.api.auth.AuthTypes;
import net.bluemind.core.api.fault.ErrorCode;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.container.service.internal.RBACManager;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.sessions.Sessions;
import net.bluemind.directory.api.IDirectory;
import net.bluemind.domain.api.Domain;
import net.bluemind.domain.api.IInCoreDomains;
import net.bluemind.keycloak.utils.sessions.JwtManager;
import net.bluemind.keydb.sessiondata.SessionData;
import net.bluemind.keydb.sessiondata.SessionDataStore;
import net.bluemind.role.api.BasicRoles;

public class SessionsMgmt implements ISessionsMgmt {
	private static final Logger logger = LoggerFactory.getLogger(SessionsMgmt.class);

	private BmContext context;

	public SessionsMgmt(BmContext context) {
		this.context = context;
	}

	@Override
	public void logoutUser(String uid) {
		RBACManager.forContext(context).check(BasicRoles.ROLE_SYSTEM_MANAGER);

		List<SecurityContext> userSessions = Sessions.get().asMap().values().stream()
				.filter(sc -> sc.getSubject().equals(uid)).toList();

		if (userSessions.isEmpty()) {
			return;
		}

		AuthTypes domainAuthType = getUserDomainAuthType(userSessions.get(0).getContainerUid());

		userSessions.forEach(us -> invalidateSession(domainAuthType, us));
	}

	private AuthTypes getUserDomainAuthType(String domainUid) {
		ItemValue<Domain> userDomain = context.provider().instance(IInCoreDomains.class).getUnfiltered(domainUid);

		if (userDomain == null) {
			logger.error("Domain {} not found !?", domainUid);
			throw new ServerFault("Domain " + domainUid + " not found !?", ErrorCode.NOT_FOUND);
		}

		return AuthTypes.get(userDomain.value.properties.get(AuthDomainProperties.AUTH_TYPE.name()));
	}

	private void invalidateSession(AuthTypes domainAuthType, SecurityContext sc) {
		if (sc.getSessionId() == null) {
			return;
		}

		if (logger.isDebugEnabled()) {
			logger.debug("logout user {} session {}", sc.getSubject(), sc.getSessionId());
		}

		SessionData sessionData = SessionDataStore.get().getIfPresent(sc.getSessionId());
		if (!domainAuthType.useBlueMindKeycloak() || sessionData == null || sessionData.jwtToken == null) {
			// If domain don't use internal authentication server or no session data or
			// jwtToken found, manage only core session...
			Sessions.get().invalidate(sc.getSessionId());
			return;
		}

		// Logout session via internal authentication server and back channel logout
		JwtManager.fromJson(sessionData.realm, sessionData.jwtToken).ifPresent(JwtManager::logoutUserSession);
	}

	@Override
	public List<SessionEntry> list(String domainUid) {
		RBACManager.forContext(context).check(BasicRoles.ROLE_SYSTEM_MANAGER);
		return Sessions.get().asMap().values().stream()
				.filter(v -> domainUid != null ? v.getContainerUid().equals(domainUid) : true)
				.map(sc -> SessionEntry.build(sc.getCreated(),
						Optional.ofNullable(context.provider().instance(IDirectory.class, sc.getContainerUid())
								.findByEntryUid(sc.getSubject())).map(de -> de.email).orElse(null),
						sc.getContainerUid(), sc.getSubject(), sc.getOrigin(), sc.getRemoteAddresses()))
				.toList();
	}

	@Override
	public void updateCurrent(SessionUpdate ud) {
		RBACManager.forContext(context).checkNotAnoynmous();
		String sid = context.getSecurityContext().getSessionId();
		if (!context.getSecurityContext().fromGlobalVirt() && sid != null) {
			SecurityContext current = Sessions.get().getIfPresent(sid);
			Sessions.get().put(sid, current.from(ud.remoteIps));
			logger.debug("Remote IPs updated to {}", ud.remoteIps);
		}
	}
}
