/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.authentication.handler;

import java.security.InvalidParameterException;
import java.util.Base64;
import java.util.Iterator;
import java.util.Optional;
import java.util.concurrent.ExecutorService;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Splitter;

import io.vertx.core.Handler;
import io.vertx.core.MultiMap;
import io.vertx.core.Vertx;
import io.vertx.core.http.HttpServerRequest;
import io.vertx.core.http.HttpServerResponse;
import net.bluemind.authentication.api.IAuthentication;
import net.bluemind.authentication.api.ValidationKind;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.core.rest.http.vertx.NeedVertxExecutor;
import net.bluemind.domain.api.Domain;
import net.bluemind.domain.api.IDomains;
import net.bluemind.hornetq.client.MQ.SharedMap;
import net.bluemind.hornetq.client.Shared;
import net.bluemind.lib.vertx.utils.PasswordDecoder;
import net.bluemind.network.topology.IServiceTopology;
import net.bluemind.network.topology.Topology;
import net.bluemind.server.api.TagDescriptor;
import net.bluemind.system.api.SysConfKeys;
import net.bluemind.user.api.IUser;
import net.bluemind.user.api.User;

public final class Nginx implements Handler<HttpServerRequest>, NeedVertxExecutor {
	private static final Logger logger = LoggerFactory.getLogger(Nginx.class);
	private Vertx vertx;

	private static class QueryParameters {
		public final String clientIp;
		public final String backendPort;
		public final String protocol;
		public final String password;
		public final String user;
		public final String latd;
		public final long time;
		public final int attempt;

		private static final SharedMap<String, String> sharedMap = Shared.mapSysconf();

		private static final String getDefaultDomain() {
			return sharedMap.get(SysConfKeys.default_domain.name());
		}

		private QueryParameters(String clientIp, String protocol, String user, String latd, String password,
				String backendPort, long time, int attempt) {
			this.clientIp = clientIp;
			this.protocol = protocol;
			this.user = user;
			this.latd = latd;
			this.password = password;
			this.backendPort = backendPort;
			this.time = time;
			this.attempt = attempt;
		}

		public static QueryParameters fromRequest(HttpServerRequest req, long time) {
			String clientIp = req.headers().get("Client-IP");
			String backendPort = req.headers().get("X-Auth-Port");
			String protocol = req.headers().get("Auth-Protocol");
			int attempt = Optional.ofNullable(req.headers().get("Auth-Login-Attempt")).map(Integer::parseInt).orElse(0);

			String user = req.headers().get("Auth-User");
			if (user == null || "".equals(user)) {
				throw new InvalidParameterException("null or empty login");
			}

			user = new String(decode(user)).toLowerCase();
			String defaultDomain = getDefaultDomain();
			String latd = (!"admin0".equals(user) && defaultDomain != null && !user.contains("@"))
					? user + "@" + defaultDomain
					: user;

			String password = PasswordDecoder.getPassword(user, decode(req.headers().get("Auth-Pass")));
			if (logger.isDebugEnabled()) {
				logger.debug("Password b64: {}, decoded: {}", req.headers().get("Auth-Pass"), password);
			}

			return new QueryParameters(clientIp, protocol, user, latd, password, backendPort, time, attempt);
		}

	}

	private static class AuthResponse {
		ValidationKind validation;
		String backendSrv;
		String backendLatd;

		public static AuthResponse of(ValidationKind kind, String backendLatd, String backendSrv) {
			AuthResponse ar = new AuthResponse();
			ar.validation = kind;
			ar.backendLatd = backendLatd;
			ar.backendSrv = backendSrv;
			return ar;
		}
	}

	public Nginx() {
	}

	@Override
	public void handle(final HttpServerRequest req) {
		long time = System.currentTimeMillis();
		req.endHandler(v -> {
			HttpServerResponse resp = req.response();
			if (!Topology.getIfAvailable().isPresent() || vertx == null) {
				resp.setStatusCode(503).setStatusMessage("Topology still missing").end();
				return;
			}
			QueryParameters qp = QueryParameters.fromRequest(req, time);

			vertx.executeBlocking(() -> this.computeResponse(qp), false) //
					.onSuccess(ar -> {
						if (ar.validation == ValidationKind.NONE || ar.validation == ValidationKind.PASSWORDEXPIRED) {
							fail(qp, resp);
						} else {
							succeed(resp, qp, ar.backendSrv, ar.backendLatd);
						}
						resp.end();
					}) //
					.onFailure(ex -> {
						logger.error(ex.getMessage(), ex);
						fail(qp, resp);
						resp.end();
					});

		});
	}

	private AuthResponse computeResponse(QueryParameters qp) {
		IAuthentication authApi = ServerSideServiceProvider.getProvider(SecurityContext.ANONYMOUS)
				.instance(IAuthentication.class);
		ValidationKind kind = authApi.validate(qp.latd, qp.password, "nginx-imap-password-check");
		if (kind != ValidationKind.NONE && kind != ValidationKind.PASSWORDEXPIRED) {
			if (!qp.latd.contains("@")) {
				throw new InvalidParameterException("Invalid login@domain " + qp.latd);
			}

			Splitter splitter = Splitter.on('@').omitEmptyStrings().trimResults();
			Iterator<String> parts = splitter.split(qp.latd).iterator();
			parts.next();
			String domainPart = parts.next();
			ServerSideServiceProvider provider = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM);

			ItemValue<Domain> domain = provider.instance(IDomains.class).findByNameOrAliases(domainPart);

			if (domain == null) {
				throw new InvalidParameterException("Fail to find domain " + domainPart);
			}

			IUser userApi = provider.instance(IUser.class, domain.uid);
			ItemValue<User> user = userApi.byEmail(qp.latd);
			if (user == null) {
				// User latd exists but email is given to something else (mailshare, group...)
				return AuthResponse.of(ValidationKind.NONE, null, null);
			}

			String backendLatd = user.value.login + "@" + domain.value.name;

			String backendSrv = null;
			IServiceTopology topology = Topology.get();
			if (topology.singleNode()) {
				backendSrv = topology.core().value.address();
			} else {
				backendSrv = topology.any(TagDescriptor.bm_core.getTag()).value.address();
			}

			long time = System.currentTimeMillis() - qp.time;
			logger.info("[name={};protocol={},oip={};backend={}] resolved in {}ms.", qp.latd, qp.protocol, qp.clientIp,
					backendSrv, time);
			return AuthResponse.of(kind, backendLatd, backendSrv);
		}

		return AuthResponse.of(kind, null, null);
	}

	/**
	 * @param latd
	 * @param resp
	 */
	private void fail(QueryParameters qp, HttpServerResponse resp) {
		logger.error("[{}] Denied auth from {}", qp == null ? null : qp.latd, qp == null ? null : qp.clientIp);
		resp.headers().add("Auth-Status", "Invalid login or password");
		if (qp != null && qp.attempt < 10) {
			resp.headers().add("Auth-Wait", "2");
		}
	}

	private void succeed(HttpServerResponse resp, QueryParameters qp, String backendSrv, String backendLatd) {
		MultiMap respHeaders = resp.headers();

		respHeaders.add("Auth-Status", "OK");
		respHeaders.add("Auth-Server", backendSrv);
		respHeaders.add("Auth-Port", qp.backendPort);

		// Support for login without @domain.tld and alias
		if (!qp.user.equals(backendLatd) || !qp.latd.equals(backendLatd)) {
			respHeaders.add("Auth-User", backendLatd);
		}
	}

	/**
	 * @param b64
	 * @return
	 */
	public static byte[] decode(String b64) {
		return Base64.getDecoder().decode(b64);
	}

	@Override
	public void setVertxExecutor(Vertx vertx, ExecutorService bmExecutor) {
		this.vertx = vertx;
	}
}
