/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2023
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.addressbook.service.internal;

import java.sql.SQLException;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import net.bluemind.addressbook.api.VCard;
import net.bluemind.addressbook.api.VCard.Kind;
import net.bluemind.addressbook.persistence.VCardIndexStore;
import net.bluemind.addressbook.persistence.VCardStore;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.Container;
import net.bluemind.core.container.model.Item;
import net.bluemind.core.container.model.ItemFlag;
import net.bluemind.core.container.model.ItemFlagFilter;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.container.repository.ICustomDecorator;
import net.bluemind.core.container.repository.IItemValueStore;
import net.bluemind.core.container.service.internal.ContainerStoreService;
import net.bluemind.core.container.service.internal.ItemValueAuditLogService;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.tx.wrapper.TxEnabler;
import net.bluemind.document.storage.DocumentStorage;
import net.bluemind.document.storage.IDocumentStore;
import net.bluemind.lib.vertx.VertxPlatform;
import net.bluemind.system.api.SystemState;
import net.bluemind.system.state.StateContext;
import net.bluemind.tag.api.TagRef;
import net.bluemind.tag.service.IInCoreTagRef;

public class VCardContainerStoreService extends ContainerStoreService<VCard> {

	private IDocumentStore documentStore;
	private IInCoreTagRef tagRefService;
	VCardIndexStore indexStore;
	private AddressBookEventProducer eventProducer;

	public VCardContainerStoreService(BmContext context, SecurityContext securityContext, Container container,
			IItemValueStore<VCard> itemValueStore, VCardIndexStore indexStore,
			ItemValueAuditLogService<VCard> logService) {
		super(context, container, itemValueStore, card -> ItemFlag.SEEN, c -> 0L, s -> s, logService);
		tagRefService = context.su().provider().instance(IInCoreTagRef.class, container.uid);
		documentStore = DocumentStorage.store;
		this.eventProducer = new AddressBookEventProducer(container, securityContext, VertxPlatform.eventBus());
		this.indexStore = indexStore;
	}

	@Override
	protected void decorate(List<Item> items, List<ItemValue<VCard>> values) throws ServerFault {
		if (StateContext.getState() == SystemState.CORE_STATE_CLONING) {
			return;
		}
		List<List<TagRef>> refs;
		try {
			refs = tagRefService.getMultiple(items);
		} catch (ServerFault sf) {
			logger.error(sf.getMessage(), sf);
			return;
		}

		Iterator<Item> itItems = items.iterator();
		Iterator<ItemValue<VCard>> itValues = values.iterator();
		Iterator<List<TagRef>> itRefs = refs.iterator();
		while (itItems.hasNext()) {
			Item item = itItems.next();
			ItemValue<VCard> value = itValues.next();
			List<TagRef> ref = itRefs.next();
			if (ref != null) {
				value.value.explanatory.categories = ref;
			}
			value.value.identification.photo = hasPhoto(item.uid);
		}

	}

	@Override
	protected void decorate(Item item, ItemValue<VCard> value, List<? extends ICustomDecorator> customDecorator) {
		if (StateContext.getState() == SystemState.CORE_STATE_CLONING) {
			return;
		}
		if (value.value == null) {
			return;
		}

		try {
			value.value.explanatory.categories = tagRefService.get(item);
		} catch (ServerFault sf) {
			logger.error(sf.getMessage(), sf);
		}

		value.value.identification.photo = AddressBookService.CustomDecorator.PHOTO.shouldApply(customDecorator)
				&& hasPhoto(item.uid);
	}

	@Override
	protected void decorate(Item item, ItemValue<VCard> value) throws ServerFault {
		if (StateContext.getState() == SystemState.CORE_STATE_CLONING) {
			return;
		}
		if (value.value == null) {
			return;
		}

		try {
			value.value.explanatory.categories = tagRefService.get(item);
		} catch (ServerFault sf) {
			logger.error(sf.getMessage(), sf);
		}

		value.value.identification.photo = hasPhoto(item.uid);

	}

	@Override
	protected void createValue(Item item, VCard value, IItemValueStore<VCard> itemValueStore)
			throws ServerFault, SQLException {
		super.createValue(item, value, itemValueStore);
		List<TagRef> tags = value.explanatory.categories;
		if (tags == null) {
			tags = Collections.emptyList();
		}
		tagRefService.create(item, tags);
		TxEnabler.durableStorageAction(() -> indexStore.create(item, value).whenComplete((ir, t) -> {
			if (t != null) {
				logger.error("index.create failed", t);
			}
			eventProducer.vcardCreated(item.uid);
			eventProducer.changed();
		}));
	}

	@Override
	protected void updateValue(Item item, VCard value) throws ServerFault, SQLException {
		super.updateValue(item, value);
		List<TagRef> tags = value.explanatory.categories;
		if (tags == null) {
			tags = Collections.emptyList();
		}
		tagRefService.update(item, tags);

		TxEnabler.durableStorageAction(() -> indexStore.update(item, value).whenComplete((ir, t) -> {
			if (t != null) {
				logger.error("index.update failed", t);
			} else {
				eventProducer.vcardUpdated(item.uid);
				eventProducer.changed();
			}
		}));
	}

	@Override
	protected void deleteValue(Item item) throws ServerFault, SQLException {
		super.deleteValue(item);
		tagRefService.delete(item);
		TxEnabler.durableStorageAction(() -> {
			if (documentStore.exists(getPhotoUid(item.uid))) {
				documentStore.delete(getPhotoUid(item.uid));
			}

			if (documentStore.exists(getIconUid(item.uid))) {
				documentStore.delete(getIconUid(item.uid));
			}
			indexStore.delete(item.uid).whenComplete((dr, t) -> {
				eventProducer.vcardDeleted(item.uid);
				eventProducer.changed();
			});
		});
	}

	private String getPhotoUid(String uid) {
		return "book_" + container.uid + "/photos/" + uid;
	}

	private String getIconUid(String uid) {
		return "book_" + container.uid + "/icons/" + uid;
	}

	@Override
	protected void deleteValues() throws ServerFault {
		super.deleteValues();
		tagRefService.deleteAll();
		TxEnabler.durableStorageAction(() -> indexStore.deleteAll().whenComplete((ir, t) -> {
			if (t != null) {
				logger.error("indexStore.deleteAll failed", t);
			} else {
				eventProducer.changed();
			}
		}));
	}

	public void setPhoto(String uid, byte[] photo) throws ServerFault {
		documentStore.store(getPhotoUid(uid), photo);
	}

	public void deletePhoto(String uid) throws ServerFault {
		documentStore.delete(getPhotoUid(uid));
	}

	public void deleteIcon(String uid) throws ServerFault {
		documentStore.delete(getIconUid(uid));
	}

	public void setIcon(String uid, byte[] photo) throws ServerFault {
		documentStore.store(getIconUid(uid), photo);
	}

	public byte[] getPhoto(String uid) throws ServerFault {
		return documentStore.get(getPhotoUid(uid));
	}

	public boolean hasPhoto(String uid) throws ServerFault {
		return documentStore.exists(getPhotoUid(uid));
	}

	public byte[] getIcon(String uid) throws ServerFault {
		return documentStore.get(getIconUid(uid));
	}

	public List<String> findByEmail(String email) throws ServerFault {
		try {
			return ((VCardStore) itemValueStore).findByEmail(email);
		} catch (SQLException e) {
			throw ServerFault.sqlFault(e);
		}
	}

	public List<String> findGroupsContaining(String[] uid) {
		try {
			return ((VCardStore) itemValueStore).findGroupsContaining(uid);
		} catch (SQLException e) {
			throw ServerFault.sqlFault(e);
		}
	}

	public List<String> findGroups() {
		try {
			return ((VCardStore) itemValueStore).findByKind(Kind.group);
		} catch (SQLException e) {
			throw ServerFault.sqlFault(e);
		}
	}

	public long getItemCount() {
		return count(ItemFlagFilter.all()).total;
	}

}
