/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */

package net.bluemind.addressbook.adapter;

import java.io.StringReader;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Optional;
import java.util.function.Function;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.addressbook.api.VCard;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.utils.UIDGenerator;
import net.bluemind.directory.api.BaseDirEntry.Kind;
import net.bluemind.tag.api.TagRef;
import net.fortuna.ical4j.vcard.Property;
import net.fortuna.ical4j.vcard.Property.Id;
import net.fortuna.ical4j.vcard.property.Version;

public final class VCardAdapter {

	private static final Logger LOGGER = LoggerFactory.getLogger(VCardAdapter.class);

	public static final List<ItemValue<VCard>> parseBmVCards(String vcard, String domainUid, String owner, Kind kind) {
		Optional<AddressbookOwner> ownerAb = Optional.of(new AddressbookOwner(domainUid, owner, kind));
		return parse(vcard).stream().map(item -> adaptCardToBm(item, s -> s, ownerAb, Collections.emptyList()))
				.toList();
	}

	public static final List<net.fortuna.ical4j.vcard.VCard> parse(String vcard) {
		List<net.fortuna.ical4j.vcard.VCard> ret = new LinkedList<>();
		try (ProgressiveVCardBuilder builder = new ProgressiveVCardBuilder(new StringReader(vcard))) {
			while (builder.hasNext()) {
				net.fortuna.ical4j.vcard.VCard card = builder.next();
				if (card != null) {
					ret.add(card);
				}
			}
		} catch (Exception e) {
			LOGGER.warn("Cannot parse vcard string", e);
		}
		return ret;
	}

	public static final ItemValue<VCard> adaptCardToBm(net.fortuna.ical4j.vcard.VCard card,
			Function<String, String> uidGenerator, Optional<AddressbookOwner> addressbookOwner, List<TagRef> allTags) {
		String retUid = UIDGenerator.uid();
		VCardAdapterToBm propertyAdapter = new VCardAdapterToBm(card, uidGenerator);
		VCard retCard = propertyAdapter.getRetCard();

		Property uidProp = card.getProperty(Id.UID);
		if (uidProp != null) {
			String value = uidProp.getValue();
			if (value.contains(",")) {
				retUid = uidGenerator.apply(value.split(",")[1]);
			}
		}

		propertyAdapter.setIdentification();
		propertyAdapter.setDeliveryAddressing();
		propertyAdapter.setCategories(addressbookOwner, allTags);
		propertyAdapter.setCommunications();
		propertyAdapter.setOrganization();
		propertyAdapter.setExplanatory();
		propertyAdapter.setSecurityKeys();
		propertyAdapter.setGroupMembers();

		return ItemValue.create(retUid, retCard);
	}

	public static final net.fortuna.ical4j.vcard.VCard adaptCardFromBm(String containerUid, VCard vcard,
			VCardVersion version) {

		VCardAdapterFromBm propertyAdapter = new VCardAdapterFromBm(vcard);

		propertyAdapter.add(new Version("3.0"));
		propertyAdapter.addKind(version);
		propertyAdapter.addIdentification();
		propertyAdapter.addDeliveringAddressing();
		propertyAdapter.addCategories();
		propertyAdapter.addCommunications();
		propertyAdapter.addOrganization();
		propertyAdapter.addExplanatory();
		propertyAdapter.addSecurityKeys();
		propertyAdapter.addGroupMembers(version, containerUid);

		return propertyAdapter.getRet();
	}

}
