/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.addressbook.adapter;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Strings;

import net.bluemind.lib.ical4j.vcard.Builder;
import net.fortuna.ical4j.data.ParserException;
import net.fortuna.ical4j.data.UnfoldingReader;
import net.fortuna.ical4j.vcard.VCard;
import net.fortuna.ical4j.vcard.VCardBuilder;

public class ProgressiveVCardBuilder implements Iterator<VCard>, AutoCloseable {

	private static final Logger logger = LoggerFactory.getLogger(ProgressiveVCardBuilder.class);
	private BufferedReader reader;

	private boolean endOfFile = false;
	private StringBuilder currentElement;
	private static final String CRLF = "\r\n";
	private static final String HTML_MULTI_LINE = "<br/>";
	private static List<String> validProperties = new ArrayList<>();

	static {
		validProperties
				.addAll(Arrays.asList(net.fortuna.ical4j.vcard.Property.Id.values()).stream().map(Enum::name).toList());
		validProperties.add("BEGIN");
		validProperties.add("END");
	}

	static final Pattern PROPERTY_NAME_PATTERN = Pattern
			.compile("^(([a-zA-Z-\\d]+\\.)?[a-zA-Z]+(?=[;:]))|" + "(([a-zA-Z-\\d]+\\.)?[Xx]-[a-zA-Z-]+(?=[;:]))");

	public ProgressiveVCardBuilder(Reader reader) {
		this.reader = new BufferedReader(reader);
		currentElement = new StringBuilder();
	}

	public VCard next() {
		if (!hasNext()) {
			throw new NoSuchElementException();
		}
		try {
			return nextImpl();
		} catch (IOException | ParserException e) {
			logger.warn("Cannot parse vcard stream", e);
			throw new RuntimeException(e);
		}
	}

	private VCard nextImpl() throws IOException, ParserException {

		boolean endOfCard = false;
		String line = null;
		StringBuilder property = new StringBuilder();
		String previousProp = null;

		while (!endOfCard && ((line = reader.readLine()) != null)) {
			// Yahoo! Crap vcard workaround.
			// SOURCE:Yahoo! AddressBook (http://address.yahoo.com) =>
			// invalid
			// see http://tools.ietf.org/html/rfc2425#section-6.1
			//
			// REV;CHARSET=utf-8:53 => invalid
			// see http://tools.ietf.org/html/rfc6350#section-6.7.4
			if (line.toUpperCase().startsWith("SOURCE") || line.toUpperCase().startsWith("REV")) {
				continue;
			}
			String data = line.replace("\\:", ":");

			boolean isNewPropertyLine = isValidNewLineProperty(data);

			// property next to => add previous property
			if (isNewPropertyLine && previousProp != null) {
				currentElement.append(previousProp).append(CRLF);
			}

			if (isNewPropertyLine) {
				// new property
				property = new StringBuilder();
				property.append(data);
			} else {
				if (!(data.startsWith(" ") || data.startsWith("\t"))) {
					// property multi line
					property.append(HTML_MULTI_LINE).append(data);
				} else {
					property.append(CRLF).append(data);
				}
			}

			previousProp = property.toString();
			endOfCard = data.toUpperCase().startsWith("END:VCARD");
			if (endOfCard) {
				currentElement.append(property.toString()).append(CRLF);
			}
		}

		String asString = currentElement.toString();
		if (Strings.isNullOrEmpty(line)) {
			endOfFile = true;
		} else {
			String lookAhead = reader.readLine();
			if (Strings.isNullOrEmpty(lookAhead)) {
				endOfFile = true;
			} else {
				currentElement.setLength(0);
				currentElement.append(lookAhead).append(CRLF);
			}
		}

		if (asString.trim().isBlank()) {
			return null;
		}

		byte[] bytes = asString.getBytes();
		VCardBuilder builder = Builder
				.from(new UnfoldingReader(new InputStreamReader(new ByteArrayInputStream(bytes)), bytes.length, true));
		return builder.buildAll().get(0);
	}

	private boolean isValidNewLineProperty(String data) {
		Matcher matcher = PROPERTY_NAME_PATTERN.matcher(data);
		if (matcher.find()) {
			String propValue = matcher.group().toUpperCase();
			return validProperties.stream()
					.anyMatch(validProp -> propValue.equals(validProp) || propValue.startsWith("X-"));
		}
		return false;
	}

	@Override
	public void close() throws Exception {
		reader.close();
	}

	@Override
	public boolean hasNext() {
		return !endOfFile;
	}

}
