/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2024
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.addressbook.adapter;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Base64;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;
import java.util.function.Function;
import java.util.stream.Collectors;

import org.apache.james.mime4j.dom.address.Mailbox;
import org.apache.james.mime4j.field.address.LenientAddressBuilder;
import org.slf4j.LoggerFactory;

import com.google.common.base.Joiner;
import com.google.common.base.Strings;

import net.bluemind.addressbook.api.VCard;
import net.bluemind.addressbook.api.VCard.Communications;
import net.bluemind.addressbook.api.VCard.Communications.Tel;
import net.bluemind.addressbook.api.VCard.DeliveryAddressing;
import net.bluemind.addressbook.api.VCard.Parameter;
import net.bluemind.core.api.Regex;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.directory.api.BaseDirEntry;
import net.bluemind.tag.api.ITagUids;
import net.bluemind.tag.api.ITags;
import net.bluemind.tag.api.Tag;
import net.bluemind.tag.api.TagRef;
import net.bluemind.utils.SyncHttpClient;
import net.fortuna.ical4j.vcard.ParameterFactory;
import net.fortuna.ical4j.vcard.ParameterFactoryRegistry;
import net.fortuna.ical4j.vcard.Property;
import net.fortuna.ical4j.vcard.Property.Id;
import net.fortuna.ical4j.vcard.property.Address;
import net.fortuna.ical4j.vcard.property.BDay;
import net.fortuna.ical4j.vcard.property.Email;
import net.fortuna.ical4j.vcard.property.Gender;
import net.fortuna.ical4j.vcard.property.Impp;
import net.fortuna.ical4j.vcard.property.Key;
import net.fortuna.ical4j.vcard.property.N;
import net.fortuna.ical4j.vcard.property.Nickname;
import net.fortuna.ical4j.vcard.property.Note;
import net.fortuna.ical4j.vcard.property.Org;
import net.fortuna.ical4j.vcard.property.Photo;
import net.fortuna.ical4j.vcard.property.Role;
import net.fortuna.ical4j.vcard.property.Telephone;
import net.fortuna.ical4j.vcard.property.Title;
import net.fortuna.ical4j.vcard.property.Url;

/**
 * Adapter from {@link net.fortuna.ical4j.vcard.VCard} (ical4j) {@link VCard}
 * (BM api)
 */
public class VCardAdapterToBm {

	private static final ParameterFactoryRegistry parameterFactoryRegistry;
	private static final Joiner join = Joiner.on(',').skipNulls();
	private static final String containerSeparator = "#";

	private final net.fortuna.ical4j.vcard.VCard card;
	private final Function<String, String> uidGenerator;
	private VCard retCard;

	private static final Map<String, String> KNOWN_TYPES = new HashMap<>();

	static {
		KNOWN_TYPES.put("home", "home");
		KNOWN_TYPES.put("work", "work");
		KNOWN_TYPES.put("text", "text");
		KNOWN_TYPES.put("voice", "voice");
		KNOWN_TYPES.put("fax", "fax");
		KNOWN_TYPES.put("cell", "cell");
		KNOWN_TYPES.put("video", "video");
		KNOWN_TYPES.put("pager", "pager");
		KNOWN_TYPES.put("ttytdd", "textphone");
		parameterFactoryRegistry = new ParameterFactoryRegistry();
		parameterFactoryRegistry.register("EXTENDED", new ExtendedFactory());
	}

	public VCardAdapterToBm(net.fortuna.ical4j.vcard.VCard card, Function<String, String> uidGenerator) {
		this.card = card;
		this.uidGenerator = uidGenerator;
		retCard = new VCard();
	}

	public VCard getRetCard() {
		return retCard;
	}

	public void setIdentification() {
		setFormattedName();
		setName();
		setNickname();
		setBirthday();
		setGender();
		setPhoto();
	}

	private void setFormattedName() {
		Property fn = card.getProperty(Id.FN);
		if (fn != null) {
			retCard.identification.formatedName = VCard.Identification.FormatedName.create(fn.getValue(),
					fromVCard(fn.getParameters()));
		}
	}

	private void setName() {
		N name = (N) card.getProperty(Id.N);
		if (name != null) {
			retCard.identification.name = VCard.Identification.Name.create(name.getFamilyName(), name.getGivenName(),
					asString(name.getAdditionalNames()), asString(name.getPrefixes()), asString(name.getSuffixes()),
					fromVCard(name.getParameters()));
		}
	}

	private void setNickname() {
		Nickname nn = (Nickname) card.getProperty(Id.NICKNAME);
		if (nn != null) {
			retCard.identification.nickname = VCard.Identification.Nickname.create(nn.getValue());
		}
	}

	private void setBirthday() {
		BDay bday = (BDay) card.getProperty(Id.BDAY);
		if (bday != null) {
			retCard.identification.birthday = bday.getDate();
		}
	}

	private void setGender() {
		Gender gender = (Gender) card.getProperty(Id.GENDER);
		if (gender != null) {
			retCard.identification.gender = VCard.Identification.Gender.create(gender.getValue(), null);
		}
	}

	private void setPhoto() {
		Photo photo = (Photo) card.getProperty(Id.PHOTO);
		if (photo != null) {
			retCard.identification.photoBinary = photo.getBinary();
		}
	}

	public void setDeliveryAddressing() {
		List<Property> props = card.getProperties(Id.ADR);
		List<DeliveryAddressing> das = new ArrayList<>(props.size());
		for (Property p : props) {
			Address adr = (Address) p;
			net.fortuna.ical4j.vcard.Parameter valueParam = p.getParameter(net.fortuna.ical4j.vcard.Parameter.Id.VALUE);
			String value = null;
			if (valueParam != null) {
				value = valueParam.getValue();
			}
			das.add(VCard.DeliveryAddressing.create(VCard.DeliveryAddressing.Address.create(value, adr.getPoBox(),
					adr.getExtended(), adr.getStreet(), adr.getLocality(), adr.getRegion(), adr.getPostcode(),
					adr.getCountry(), fromVCard(adr.getParameters()))));
		}

		retCard.deliveryAddressing = das;
	}

	public void setCommunications() {
		setTels();
		setEmails();
		setImpps();
	}

	private void setTels() {
		List<Property> telProps = card.getProperties(Id.TEL);
		List<Tel> tels = new ArrayList<>(telProps.size());
		for (Property p : telProps) {
			Telephone tel = (Telephone) p;

			tels.add(Tel.create(tel.getValue(), fromVCard(tel.getParameters())));
		}

		retCard.communications.tels = tels;
	}

	private void setEmails() {
		List<Property> mailProps = card.getProperties(Id.EMAIL);
		List<Communications.Email> mails = new ArrayList<>(mailProps.size());
		for (Property p : mailProps) {
			Email mail = (Email) p;
			String emailValue = mail.getValue();
			if (!Regex.EMAIL.validate(emailValue)) {
				emailValue = Optional.ofNullable(LenientAddressBuilder.DEFAULT.parseMailbox(emailValue))
						.map(Mailbox::getAddress).orElse(emailValue);
				// if everything fail, we import email as it (because rfc
				// authorize it)
			}
			mails.add(Communications.Email.create(emailValue, fromVCard(mail.getParameters())));
		}
		retCard.communications.emails = mails;
	}

	private void setImpps() {
		List<Property> imppProps = card.getProperties(Id.IMPP);
		List<Communications.Impp> impps = new ArrayList<>(imppProps.size());
		for (Property p : imppProps) {
			Impp impp = (Impp) p;
			impps.add(Communications.Impp.create(impp.getValue(), fromVCard(impp.getParameters())));
		}
		retCard.communications.impps = impps;
	}

	public void setExplanatory() {
		setUrls();
		setNote();
	}

	private void setUrls() {
		List<Property> urlProps = card.getProperties(Id.URL);
		List<VCard.Explanatory.Url> urls = new ArrayList<>(urlProps.size());
		for (Property p : urlProps) {
			String urlValue = null;
			List<net.fortuna.ical4j.vcard.Parameter> params = null;
			if (p instanceof net.bluemind.lib.ical4j.vcard.property.Url) {
				net.bluemind.lib.ical4j.vcard.property.Url url = (net.bluemind.lib.ical4j.vcard.property.Url) p;
				urlValue = url.getValue();
				params = url.getParameters();
			} else {
				Url url = (Url) p;
				urlValue = url.getValue();
				params = url.getParameters();
			}
			urls.add(VCard.Explanatory.Url.create(urlValue, fromVCard(params)));
		}
		retCard.explanatory.urls = urls;

	}

	private void setNote() {
		Note noteProp = (Note) card.getProperty(Id.NOTE);
		if (noteProp != null) {
			Property noteAsHtml = card.getExtendedProperty("NOTE-HTML");
			if (noteAsHtml != null) {
				retCard.explanatory.note = noteAsHtml.getValue();
			} else {
				retCard.explanatory.note = noteProp.getValue();
			}
		}
	}

	public void setCategories(Optional<AddressbookOwner> addressbookOwner, List<TagRef> allTags) {
		retCard.explanatory.categories = parseVcfCategories(card.getProperties(Id.CATEGORIES), addressbookOwner,
				allTags);
	}

	public void setSecurityKeys() {
		List<Property> keys = card.getProperties(Id.KEY);
		retCard.security.keys = new ArrayList<>();
		for (Property key : keys) {
			if (key instanceof Key) {
				Key asKey = (Key) key;
				if (asKey.getBinary() != null) {
					String decoded = null;
					try {
						decoded = new String(Base64.getDecoder().decode(key.getValue()));
					} catch (Exception e) {
						decoded = key.getValue();
					}

					retCard.security.keys.add(VCard.Security.Key.create(decoded, fromVCard(key.getParameters())));
				} else {
					String url = asKey.getValue();
					try {
						String downloaded = SyncHttpClient.getInstance().get(url);
						List<Parameter> params = fromVCard(key.getParameters()).stream()
								.filter(p -> !(p.label.equals("VALUE") && p.value.equalsIgnoreCase("url"))).toList();
						retCard.security.keys.add(VCard.Security.Key.create(downloaded, params));
					} catch (Exception e) {
						LoggerFactory.getLogger(VCardAdapter.class).warn("Cannot download contact certificate from {}",
								url, e);
					}
				}
			} else {
				retCard.security.keys.add(VCard.Security.Key.create(key.getValue(), fromVCard(key.getParameters())));
			}
		}
	}

	public void setOrganization() {
		setRole();
		setTitle();
		setOrg();
	}

	private void setRole() {
		Role role = (Role) card.getProperty(Id.ROLE);
		if (role != null) {
			retCard.organizational.role = role.getValue();
		}
	}

	private void setTitle() {
		Title title = (Title) card.getProperty(Id.TITLE);
		if (title != null) {
			retCard.organizational.title = title.getValue();
		}
	}

	private void setOrg() {

		Org org = (Org) card.getProperty(Id.ORG);
		if (org != null) {
			String[] values = org.getValues();
			String company = null;
			String division = null;
			String department = null;
			if (values.length >= 1) {
				company = values[0];
			}
			if (values.length >= 2) {
				division = values[1];
			}
			if (values.length >= 3) {
				department = values[2];
			}
			retCard.organizational.org = VCard.Organizational.Org.create(company, division, department);
		}
	}

	public void setGroupMembers() {
		// handle X-Macintosh
		// X-ADDRESSBOOKSERVER-KIND
		Property xKind = card.getExtendedProperty("ADDRESSBOOKSERVER-KIND");
		if (xKind != null && "group".equals(xKind.getValue())) {
			retCard.kind = VCard.Kind.group;

			List<Property> members = card.getExtendedProperties("ADDRESSBOOKSERVER-MEMBER");

			retCard.organizational.member = members.stream().filter(p -> p.getValue().startsWith("urn:uuid:"))
					.map(p -> {
						String value = p.getValue().substring("urn:uuid:".length());
						String containerUid = null;
						String uid = null;
						String memberName = getParamValue(p.getParameters(), "X-CN").orElse(null);
						String memberEmail = getParamValue(p.getParameters(), "X-MAILTO").orElse(null);
						if (value.contains(containerSeparator)) {
							String[] splitted = value.split(containerSeparator);
							containerUid = splitted[0];
							uid = splitted[1];
							if (memberName == null) {
								memberName = uid;
							}
						} else {
							uid = uidGenerator.apply(value);
							if (Regex.EMAIL.validate(value) && memberEmail == null) {
								memberEmail = value;
							}
							if (memberName == null) {
								memberName = value;
							}
						}
						return VCard.Organizational.Member.create(containerUid, uid, memberName, memberEmail);
					}).collect(Collectors.toList());
		}
	}

	static List<Parameter> fromVCard(List<net.fortuna.ical4j.vcard.Parameter> parameters) {

		List<Parameter> ret = new ArrayList<>(parameters.size());
		for (net.fortuna.ical4j.vcard.Parameter p : parameters) {
			String value = p.getValue();
			if (p.getId().getPname().equals("TYPE")) {
				for (String v : value.split(",")) {
					if (KNOWN_TYPES.keySet().contains(v.toLowerCase())) {
						ret.add(Parameter.create(p.getId().getPname(), KNOWN_TYPES.get(v.toLowerCase())));
					} else {
						ret.add(Parameter.create(p.getId().getPname(), v));
					}
				}
			} else {
				ret.add(Parameter.create(p.getId().getPname(), p.getValue()));
			}
		}
		return ret;
	}

	static List<net.fortuna.ical4j.vcard.Parameter> toVCard(List<Parameter> parameters) {
		List<net.fortuna.ical4j.vcard.Parameter> ret = new ArrayList<>(parameters.size());
		List<String> vcardParameters = Arrays.asList(net.fortuna.ical4j.vcard.Parameter.Id.values()).stream()
				.map(id -> id.getPname()).toList();
		parameters.stream().filter(p -> p.label != null && vcardParameters.contains(p.label)).forEach(p -> {
			ParameterFactory<?> paramFactory = parameterFactoryRegistry.getFactory(p.label);
			if (paramFactory != null) {
				ret.add(paramFactory.createParameter(p.label, p.value));
			}
		});
		return ret;
	}

	private static String asString(String[] values) {
		if (values == null)
			return null;

		return join.join(values);
	}

	private static List<TagRef> parseVcfCategories(List<Property> categoriesPropList, Optional<AddressbookOwner> owner,
			List<TagRef> allTags) {
		if (categoriesPropList == null || categoriesPropList.isEmpty()) {
			return null;
		}
		if (!owner.isPresent()) {
			return null;
		}

		AddressbookOwner abOwner = owner.get();

		Optional<String> containerUid = abOwner.kind != BaseDirEntry.Kind.ADDRESSBOOK
				&& abOwner.kind != BaseDirEntry.Kind.RESOURCE ? Optional.of(ITagUids.defaultTags(abOwner.userUid))
						: Optional.empty();
		ITags service = null;
		Sudo sudo = null;
		List<TagRef> categories = new ArrayList<>(categoriesPropList.size());
		try {
			if (containerUid.isPresent()) {
				try {
					sudo = new Sudo(abOwner.userUid, abOwner.domainUid);
					service = ServerSideServiceProvider.getProvider(sudo.context).instance(ITags.class,
							containerUid.get());
				} catch (Exception e) {
					sudo = null;
				}
			}

			for (@SuppressWarnings("unchecked")
			Iterator<Property> it = categoriesPropList.iterator(); it.hasNext();) {
				Property category = it.next();
				String labelValue = category.getValue();
				if (Strings.isNullOrEmpty(labelValue)) {
					continue;
				}
				String[] values = labelValue.split(",");
				for (String label : values) {
					Optional<TagRef> exsistingTag = allTags.stream().filter(tag -> label.equals(tag.label)).findFirst();
					if (exsistingTag.isPresent()) {
						categories.add(exsistingTag.get());
					} else {
						// 3d98ff blue
						if (service != null) {
							String uid = UUID.randomUUID().toString();
							service.create(uid, Tag.create(label, "3d98ff"));

							TagRef tr = TagRef.create(containerUid.get(), service.getComplete(uid));
							allTags.add(tr);
							categories.add(tr);
						}
					}
				}
			}
		} finally {
			if (sudo != null) {
				sudo.close();
			}
		}
		return categories;
	}

	private static Optional<String> getParamValue(List<net.fortuna.ical4j.vcard.Parameter> list, String parameterId) {
		for (net.fortuna.ical4j.vcard.Parameter p : list) {
			if (p.getId() == net.fortuna.ical4j.vcard.Parameter.Id.EXTENDED && p.toString().contains(parameterId)) {
				return Optional.of(p.getValue());
			}
		}
		return Optional.empty();
	}

}
