export class ProgressMonitor {
    static UPLOAD = 1;
    static DOWNLOAD = 2;
    static BOTH = this.UPLOAD + this.DOWNLOAD;

    constructor() {
        this.listeners = [];
    }
    addEventListener(type, listener, options) {
        const opts = normalizeOptions(options);
        const eventListener = register(this.listeners, { type, callback: listener, options: opts });
        const mode = opts.mode & ~eventListener.options.mode;
        addListener.call(this, eventListener, mode);
        eventListener.options.mode |= mode;
    }
    removeEventListener(type, listener, options) {
        const opts = normalizeOptions(options);
        const eventListener = unregister(this.listeners, { type, callback: listener, options: opts });
        const mode = opts.mode & ~eventListener.options.mode;
        removeListener.call(this, eventListener, mode);
        eventListener.options.mode &= ~mode;
    }
}

export function setupProgressMonitor(monitor, target) {
    monitor.download = target;
    monitor.upload = target.upload;
    monitor.listeners.forEach(listener => addListener.call(monitor, listener, listener.options.mode));
}


export function tearDownProgressMonitor(monitor) {
    monitor.listeners.forEach(listener => removeListener.call(monitor, listener, listener.options.mode));
}

function normalizeOptions(options) {
    const capture = normaliseCapture(options);
    const mode = normaliseMode(options);
    return { ...options, capture, mode };
}

function find(listeners, listener) {
    return listeners.find(({ type, callback, options }) => {
        if (type === listener.type && callback === listener.callback) {
            return normaliseCapture(options) === normaliseCapture(listener.options);
        }
    });
}
function register(listeners, listener) {
    const result = find(listeners, listener);
    if (!result) {
        listener = { ...listener, options: { ...listener.options, mode: 0 } };
        listeners.push(listener);
        return listener;
    }
    result.options.once &&= listener.options.once;
    result.options.passive &&= listener.options.passive;
    result.options.signal ||= listener.options.signal;
    return result;
}
function unregister(listeners, listener) {
    const result = find(listeners, listener);
    if (!result) {
        return { ...listener, options: { ...listener.options, mode: 0 } };
    }
    if ((result.options.mode & listener.options.mode) === result.options.mode) {
        listeners.splice(listeners.indexOf(result, 1));
    }
    return result;
}

function addListener(listener, mode) {
    if (this.download && mode & ProgressMonitor.DOWNLOAD) {
        this.download.addEventListener(listener.type, listener.callback, listener.options);
    }
    if (this.upload && mode & ProgressMonitor.UPLOAD) {
        this.upload.addEventListener(listener.type, listener.callback, listener.options);
    }
}

function removeListener(listener, mode) {
    if (this.download && mode & ProgressMonitor.DOWNLOAD) {
        this.download.removeEventListener(listener.type, listener.callback, listener.options);
    }
    if (this.upload && mode & ProgressMonitor.UPLOAD) {
        this.upload.removeEventListener(listener.type, listener.callback, listener.options);
    }
}
function normaliseMode(options) {
    if (typeof options === "boolean" || !options?.mode) {
        return ProgressMonitor.DOWNLOAD;
    } else {
        return options.mode & ProgressMonitor.BOTH;
    }
}

function normaliseCapture(options) {
    if (typeof options === "boolean") {
        return options;
    } else {
        return Boolean(options?.capture);
    }
}
