import { tearDownProgressMonitor, setupProgressMonitor } from "./ProgressMonitor";

export function fetch(input, options) {
    return new Promise(function (resolve, reject) {
        const xhr = new XMLHttpRequest();

        const request = new Request(input, options);
        if (request.signal?.aborted) {
            return reject(new DOMException("Aborted", "AbortError"));
        }

        xhr.open(request.method, request.url, true);

        const abort = xhr.abort.bind(xhr);
        if (request.signal) {
            request.signal.addEventListener("abort", abort);
        }
        setupProgressMonitor(options.monitor, xhr);

        xhr.onerror = () => reject(new TypeError("Network request failed"));
        xhr.ontimeout = () => reject(new TypeError("Network request failed"));
        xhr.onabort = () => reject(new DOMException("Aborted", "AbortError"));
        xhr.onload = () => resolve(response(xhr));
        xhr.onreadystatechange = function () {
            if (xhr.readyState === XMLHttpRequest.DONE) {
                if (request.signal) {
                    request.signal.removeEventListener("abort", abort);
                }
                tearDownProgressMonitor(options.monitor);
            }
        };

        xhr.withCredentials = request.credentials === "include";
        xhr.responseType = "blob";

        request.headers.forEach(function (value, name) {
            xhr.setRequestHeader(name, value);
        });

        xhr.send(options.body);
    });
}

function response(xhr) {
    return new Response(xhr.response, {
        status: xhr.status,
        statusText: xhr.statusText,
        headers: parseHeaders(xhr.getAllResponseHeaders()),
    });
}

function parseHeaders(raw) {
    const headers = raw
        .replace(/\r?\n[\t ]+/g, " ")
        .split(/\r?\n/)
        .map((header) => {
            const [key, ...values] = header.split(":");
            return [key.trim(), values.join(":")];
        })
        .filter(([key, value]) => Boolean(key));
    return new Headers(headers);
}
