import { 
    TaskRef
} from "@bluemind/core.task.api";

import { Assignment } from "./Assignment"; 
import { CommandStatus } from "./CommandStatus"; 
import { Server } from "./Server"; 
import { 
    ItemValue
} from "@bluemind/core.container.api";
import { ApiEndPoint, EndPointMetadata, RequestOptions } from "@bluemind/api.commons";
/**
 * API for managing the servers in a BlueMind installation.
 * 
 * {@link Server} is the equivalent of Host in core_v1.
 * 
 * Server are items living in an <code>installation</code> Container. The uid of
 * the container is determined using the /etc/bm/mcast.id on your bluemind
 * installation.
 */
export interface IServer {
    /**
     * Enumerate all the servers in the installation container.
     */
    allComplete( __fetchOptions?: RequestOptions): Promise<Array<ItemValue<Server>>>;
    /**
     * Assigns a Server's tag to a domain. For example, when you assign your
     * mail/imap Server to a domain blue-mind.net a mail partition will be created
     * to hold this domain data.
     * 
     * This method stores the newly made assignment in the database and IServerHook
     * implementations will do all the system work.
     */
    assign(serverUid: string, domainUid: string, tag?: string,  __fetchOptions?: RequestOptions): Promise<void>;
    /**
     * Fetches all the servers in a domain container by its assignment
     */
    byAssignment(domainUid: string, tag?: string,  __fetchOptions?: RequestOptions): Promise<Array<string>>;
    /**
     * Creates a new server in the database with the given uid. IServerHook
     * implementations are invoked.
     */
    create(uid: string, srv?: Server,  __fetchOptions?: RequestOptions): Promise<TaskRef>;
    /**
     * Deletes a server object in the database with its uid.
     */
    delete(uid: string,  __fetchOptions?: RequestOptions): Promise<void>;
    /**
     * Fetches all the server assignments in a domain container
     */
    getAssignments(domainUid: string,  __fetchOptions?: RequestOptions): Promise<Array<Assignment>>;
    /**
     * Fetches a server object from the database with its uid.
     */
    getComplete(uid: string,  __fetchOptions?: RequestOptions): Promise<ItemValue<Server>>;
    getServerAssignments(uid: string,  __fetchOptions?: RequestOptions): Promise<Array<Assignment>>;
    /**
     * Fetches the progress of a command running in bm-node
     */
    getStatus(uid: string, ref?: string,  __fetchOptions?: RequestOptions): Promise<CommandStatus>;
    /**
     * Uses bm-node to read a file on a {@link Server} with its uid.
     * @param uid the server uid
     * @param path the absolute filename to read
     * @returns the bytes in the file
     */
    readFile(uid: string, path: string,  __fetchOptions?: RequestOptions): Promise<ArrayBuffer>;
    /**
     * tags server and return ref to Task ({@link ITask}
     */
    setTags(uid: string, tags?: Array<string>,  __fetchOptions?: RequestOptions): Promise<TaskRef>;
    /**
     * Executes a command using bm-node on the server with the given uid. Returns a
     * reference to the running command that must be used in subsequent
     * <code>getStatus</code> calls.
     */
    submit(uid: string, argv?: Array<string>,  __fetchOptions?: RequestOptions): Promise<string>;
    /**
     * Executes a command using bm-node on the server with the given uid. Returns
     * command execution output and exit code ({@link CommandStatus}).
     */
    submitAndWait(uid: string, argv?: Array<string>,  __fetchOptions?: RequestOptions): Promise<CommandStatus>;
    /**
     * Undo what assign does.
     */
    unassign(serverUid: string, domainUid: string, tag?: string,  __fetchOptions?: RequestOptions): Promise<void>;
    /**
     * Updates a server in the database with its uid. IServerHook implementations
     * are invoked.
     */
    update(uid: string, srv?: Server,  __fetchOptions?: RequestOptions): Promise<TaskRef>;
    /**
     * Uses bm-node to write a file on a {@link Server} with its uid.
     * @param uid the server uid
     * @param path the absolute filename to write
     */
    writeFile(uid: string, path: string, content?: ArrayBuffer,  __fetchOptions?: RequestOptions): Promise<void>;
}

export declare class ServerClient extends ApiEndPoint implements IServer {
    containerUid: string;
    constructor(apiKey: string, containerUid: string, base?: string);
    allComplete( __fetchOptions?: RequestOptions): Promise<Array<ItemValue<Server>>>;
    assign(serverUid: string, domainUid: string, tag?: string,  __fetchOptions?: RequestOptions): Promise<void>;
    byAssignment(domainUid: string, tag?: string,  __fetchOptions?: RequestOptions): Promise<Array<string>>;
    create(uid: string, srv?: Server,  __fetchOptions?: RequestOptions): Promise<TaskRef>;
    delete(uid: string,  __fetchOptions?: RequestOptions): Promise<void>;
    getAssignments(domainUid: string,  __fetchOptions?: RequestOptions): Promise<Array<Assignment>>;
    getComplete(uid: string,  __fetchOptions?: RequestOptions): Promise<ItemValue<Server>>;
    getServerAssignments(uid: string,  __fetchOptions?: RequestOptions): Promise<Array<Assignment>>;
    getStatus(uid: string, ref?: string,  __fetchOptions?: RequestOptions): Promise<CommandStatus>;
    readFile(uid: string, path: string,  __fetchOptions?: RequestOptions): Promise<ArrayBuffer>;
    setTags(uid: string, tags?: Array<string>,  __fetchOptions?: RequestOptions): Promise<TaskRef>;
    submit(uid: string, argv?: Array<string>,  __fetchOptions?: RequestOptions): Promise<string>;
    submitAndWait(uid: string, argv?: Array<string>,  __fetchOptions?: RequestOptions): Promise<CommandStatus>;
    unassign(serverUid: string, domainUid: string, tag?: string,  __fetchOptions?: RequestOptions): Promise<void>;
    update(uid: string, srv?: Server,  __fetchOptions?: RequestOptions): Promise<TaskRef>;
    writeFile(uid: string, path: string, content?: ArrayBuffer,  __fetchOptions?: RequestOptions): Promise<void>;
    getMetadata(): EndPointMetadata;
}
