/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.scheduledjob.scheduler.impl;

import java.util.Collections;
import java.util.Date;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.core.utils.CancellableRunnable;
import net.bluemind.scheduledjob.api.IJob;
import net.bluemind.scheduledjob.api.InProgressException;
import net.bluemind.scheduledjob.api.Job;
import net.bluemind.scheduledjob.api.JobExitStatus;
import net.bluemind.scheduledjob.api.PlanKind;
import net.bluemind.scheduledjob.scheduler.IScheduledJob;
import net.bluemind.scheduledjob.scheduler.IScheduledJobRunId;

public class JobTicker implements CancellableRunnable {

	private static final Logger logger = LoggerFactory.getLogger(JobTicker.class);
	public final IScheduledJob bj;
	public final Scheduler sch;
	public final boolean forced;
	public final String domainName;
	public final Date date;
	public final String execGroup;

	public JobTicker(IScheduledJob bjp, boolean forced, String domainName, Date cur, String groupId) {
		this.bj = bjp;
		this.sch = Scheduler.get();
		this.forced = forced;
		this.domainName = domainName;
		this.date = cur;
		this.execGroup = groupId;
	}

	@Override
	public void run() {
		try {
			sch.setActiveGroup(execGroup);
			IScheduledJobRunId slot = sch.getActiveSlot(domainName, bj.getJobId());
			if (slot == null) {
				bj.tick(sch, forced, domainName, date);
				resetJobSchedule(bj);
				finishIfNeeded(null);
			}
		} catch (InProgressException ipe) {
			logger.warn("job {}@{} already in progress.", bj.getJobId(), domainName);
		} catch (ServerFault e) {
			logger.error(e.getMessage(), e);
			finishIfNeeded(e);
		} catch (Exception t) {
			logger.error(t.getMessage(), t);
			finishIfNeeded(t);
		}

	}

	private void resetJobSchedule(IScheduledJob job) {
		if (!job.supportsScheduling()) {
			try {
				IJob jobService = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM).instance(IJob.class);
				Job storedJob = jobService.getJobFromId(job.getJobId());
				if (!storedJob.domainPlanification.isEmpty()) {
					boolean scheduled = storedJob.domainPlanification.stream()
							.anyMatch(plan -> plan.kind == PlanKind.SCHEDULED);
					if (scheduled) {
						// let the store set the default plan
						storedJob.domainPlanification = Collections.emptyList();
						jobService.update(storedJob);
					}
				}
			} catch (ServerFault e) {
				logger.warn("Cannot reset job scheduling", e);
			}
		}
	}

	private void finishIfNeeded(Throwable t) {
		RunIdImpl slot = (RunIdImpl) sch.getActiveSlot(domainName, bj.getJobId());
		sch.unregister(domainName, bj);
		if (slot != null) {
			if (t != null) {
				sch.error(slot, "en", "Job halted by exception: " + t.getMessage());
				sch.error(slot, "fr", "Tâche stoppée par une exception: " + t.getMessage());
				sch.finish(slot, JobExitStatus.FAILURE);
			} else if (slot.status == JobExitStatus.IN_PROGRESS) {
				sch.warn(slot, "en", "Job completed without proper status reporting");
				sch.warn(slot, "fr", "Tâche terminée sans avoir indiquée son statut");
				sch.finish(slot, JobExitStatus.COMPLETED_WITH_WARNINGS);
			}
		}
	}

	@Override
	public void cancel() {
		bj.cancel();
	}

}
