import Vue from "vue";
import Vue2TouchEvents from "vue2-touch-events";

import { default as AlertStore, DefaultAlert } from "@bluemind/alert.store";
import NotificationManager from "@bluemind/commons/utils/notification";
import i18n, { generateDateTimeFormats } from "@bluemind/i18n";
import injector, { inject } from "@bluemind/inject";
import router from "@bluemind/router";
import { initSentry } from "@bluemind/sentry";
import store from "@bluemind/store";
import { BmModalPlugin } from "@bluemind/ui-components";
import VueBus from "@bluemind/vue-bus";
import { extend } from "@bluemind/vuex-router";
import VueSockjsPlugin from "@bluemind/vue-sockjs";

import routes from "./routes";
import registerDependencies from "./registerDependencies";
import PreferencesStore from "./preferences/store";
import RootAppStore from "./rootAppStore";
import SettingsStore from "./settingsStore";
import MainApp from "./components/MainApp";
import Command from "../plugins/Command";

initWebApp();
initSentry(Vue);

async function initWebApp() {
    registerUserSession();
    const userSession = injector.getProvider("UserSession").get();
    registerDependencies(userSession);
    setDateTimeFormat(userSession);
    // Request permission for Notification
    NotificationManager.requestPermission();
    initStore();
    setVuePlugins(userSession);
    if (userSession.userId) {
        await store.dispatch("settings/FETCH_ALL_SETTINGS"); // needed to initialize i18n
        await store.dispatch("root-app/FETCH_ALL_APP_DATA");
    }
    Vue.component("DefaultAlert", DefaultAlert);
    router.addRoutes(routes);
    new Vue({
        el: "#app",
        i18n,
        render: h => h(MainApp),
        router,
        store
    });
}

function setVuePlugins(userSession) {
    Vue.use(VueBus, store);
    if (userSession.userId) {
        Vue.use(VueSockjsPlugin, VueBus);
    }
    Vue.use(Vue2TouchEvents, { disableClick: true });
    Vue.use(BmModalPlugin);
    Vue.use(Command);
}

function registerUserSession() {
    injector.register({
        provide: "UserSession",
        use: window.bmcSessionInfos
    });
}

function initStore() {
    extend(router, store);
    store.registerModule("alert", AlertStore);
    store.registerModule("root-app", RootAppStore);
    store.registerModule("settings", SettingsStore);
    store.registerModule("preferences", PreferencesStore);
}

function setDateTimeFormat(session) {
    inject("UserSettingsPersistence")
        .getOne(session.userId, "timeformat")
        .then(timeformat => i18n.setDateTimeFormat(generateDateTimeFormats(timeformat)));
}
