/*
 * Decompiled with CFR 0.152.
 */
package io.sentry.connection;

import io.sentry.SentryClient;
import io.sentry.connection.Connection;
import io.sentry.connection.EventSendCallback;
import io.sentry.connection.LockedDownException;
import io.sentry.connection.TooManyRequestsException;
import io.sentry.environment.SentryEnvironment;
import io.sentry.event.Event;
import io.sentry.util.Nullable;
import io.sentry.util.Util;
import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;

public class AsyncConnection
implements Connection {
    private static final Logger logger = LoggerFactory.getLogger(AsyncConnection.class);
    private static final Logger lockdownLogger = LoggerFactory.getLogger((String)(SentryClient.class.getName() + ".lockdown"));
    final ShutDownHook shutDownHook = new ShutDownHook();
    private final long shutdownTimeout;
    private final Connection actualConnection;
    private final ExecutorService executorService;
    private boolean gracefulShutdown;
    private volatile boolean closed;

    public AsyncConnection(Connection actualConnection, @Nullable ExecutorService executorService, boolean gracefulShutdown, long shutdownTimeout) {
        this.actualConnection = actualConnection;
        this.executorService = executorService == null ? Executors.newSingleThreadExecutor() : executorService;
        if (gracefulShutdown) {
            this.gracefulShutdown = gracefulShutdown;
            this.addShutdownHook();
        }
        this.shutdownTimeout = shutdownTimeout;
    }

    private void addShutdownHook() {
        Runtime.getRuntime().addShutdownHook(this.shutDownHook);
    }

    @Override
    public void send(Event event) {
        if (!this.closed) {
            this.executorService.execute(new EventSubmitter(event, MDC.getCopyOfContextMap()));
        }
    }

    @Override
    public void addEventSendCallback(EventSendCallback eventSendCallback) {
        this.actualConnection.addEventSendCallback(eventSendCallback);
    }

    @Override
    public void close() throws IOException {
        if (this.gracefulShutdown) {
            Util.safelyRemoveShutdownHook(this.shutDownHook);
            this.shutDownHook.enabled = false;
        }
        this.doClose();
    }

    private void doClose() throws IOException {
        logger.debug("Gracefully shutting down Sentry async threads.");
        this.closed = true;
        this.executorService.shutdown();
        try {
            if (this.shutdownTimeout == -1L) {
                long waitBetweenLoggingMs = 5000L;
                while (!this.executorService.awaitTermination(waitBetweenLoggingMs, TimeUnit.MILLISECONDS)) {
                    logger.debug("Still waiting on async executor to terminate.");
                }
            } else if (!this.executorService.awaitTermination(this.shutdownTimeout, TimeUnit.MILLISECONDS)) {
                logger.warn("Graceful shutdown took too much time, forcing the shutdown.");
                List<Runnable> tasks = this.executorService.shutdownNow();
                logger.warn("{} tasks failed to execute before shutdown.", (Object)tasks.size());
            }
            logger.debug("Shutdown finished.");
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            logger.warn("Graceful shutdown interrupted, forcing the shutdown.");
            List<Runnable> tasks = this.executorService.shutdownNow();
            logger.warn("{} tasks failed to execute before shutdown.", (Object)tasks.size());
        }
        finally {
            this.actualConnection.close();
        }
    }

    private final class ShutDownHook
    extends Thread {
        private volatile boolean enabled = true;

        private ShutDownHook() {
        }

        @Override
        public void run() {
            if (!this.enabled) {
                return;
            }
            SentryEnvironment.startManagingThread();
            try {
                AsyncConnection.this.doClose();
            }
            catch (IOException | RuntimeException e) {
                logger.error("An exception occurred while closing the connection.", (Throwable)e);
            }
            finally {
                SentryEnvironment.stopManagingThread();
            }
        }
    }

    private final class EventSubmitter
    implements Runnable {
        private final Event event;
        private Map<String, String> mdcContext;

        private EventSubmitter(Event event, Map<String, String> mdcContext) {
            this.event = event;
            this.mdcContext = mdcContext;
        }

        @Override
        public void run() {
            SentryEnvironment.startManagingThread();
            Map previous = MDC.getCopyOfContextMap();
            if (this.mdcContext == null) {
                MDC.clear();
            } else {
                MDC.setContextMap(this.mdcContext);
            }
            try {
                AsyncConnection.this.actualConnection.send(this.event);
            }
            catch (LockedDownException | TooManyRequestsException e) {
                logger.debug("Dropping an Event due to lockdown: " + this.event);
            }
            catch (RuntimeException e) {
                logger.error("An exception occurred while sending the event to Sentry.", (Throwable)e);
            }
            finally {
                if (previous == null) {
                    MDC.clear();
                } else {
                    MDC.setContextMap((Map)previous);
                }
                SentryEnvironment.stopManagingThread();
            }
        }
    }
}

