/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2018
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.backend.mail.replica.service.internal.repair;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.stream.Collectors;

import org.slf4j.event.Level;

import com.google.common.base.Strings;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Lists;

import net.bluemind.backend.mail.replica.api.IMailReplicaUids;
import net.bluemind.backend.mail.replica.api.MailApiHeaders;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.rest.BmContext;
import net.bluemind.directory.api.BaseDirEntry.Kind;
import net.bluemind.directory.api.DirEntry;
import net.bluemind.directory.api.MaintenanceOperation;
import net.bluemind.directory.service.IDirEntryRepairSupport;
import net.bluemind.directory.service.RepairTaskMonitor;
import net.bluemind.imap.Flag;
import net.bluemind.imap.FlagsList;
import net.bluemind.imap.IMAPException;
import net.bluemind.imap.IMAPHeaders;
import net.bluemind.imap.InternalDate;
import net.bluemind.imap.ListInfo;
import net.bluemind.imap.StoreClient;
import net.bluemind.mailbox.api.IMailboxes;
import net.bluemind.mailbox.api.Mailbox;
import net.bluemind.server.api.IServer;
import net.bluemind.server.api.Server;

public class DuplicateImapRecordsRepair implements IDirEntryRepairSupport {

	public static final MaintenanceOperation op = MaintenanceOperation.create("replication.duplicates",
			"Eliminate duplicate IMAP messages by X-Bm-Internal-Id");

	private final BmContext context;

	public DuplicateImapRecordsRepair(BmContext context) {
		this.context = context;
	}

	@Override
	public Set<MaintenanceOperation> availableOperations(Kind kind) {
		if (kind == Kind.USER || kind == Kind.MAILSHARE || kind == Kind.GROUP) {
			return ImmutableSet.of(op);
		}
		return Collections.emptySet();
	}

	@Override
	public Set<InternalMaintenanceOperation> ops(Kind kind) {
		if (kind == Kind.USER || kind == Kind.MAILSHARE || kind == Kind.GROUP) {
			return ImmutableSet.of(new DuplicateRecordMaintenance(context));
		}
		return Collections.emptySet();

	}

	public static class RepairFactory implements IDirEntryRepairSupport.Factory {
		@Override
		public IDirEntryRepairSupport create(BmContext context) {
			return new DuplicateImapRecordsRepair(context);
		}
	}

	private static class DuplicateRecordMaintenance extends InternalMaintenanceOperation {

		private final BmContext context;

		public DuplicateRecordMaintenance(BmContext ctx) {
			super(op.identifier, null, IMailReplicaUids.REPAIR_SUBTREE_OP, 1);
			this.context = ctx;
		}

		@Override
		public void check(String domainUid, DirEntry entry, RepairTaskMonitor monitor) {
			if (entry.archived) {
				monitor.end(true, "DirEntry is archived, skip it", null);
				return;
			}

		}

		@Override
		public void repair(String domainUid, DirEntry entry, RepairTaskMonitor monitor) {

			if (entry.archived) {
				monitor.end(true, "DirEntry is archived, skip it", null);
				return;
			}

			monitor.log("Repair replication duplicates {} {}", domainUid, entry);

			IMailboxes iMailboxes = context.getServiceProvider().instance(IMailboxes.class, domainUid);
			ItemValue<Mailbox> mbox = iMailboxes.getComplete(entry.entryUid);

			IServer iServer = context.getServiceProvider().instance(IServer.class, "default");
			ItemValue<Server> server = iServer.getComplete(entry.dataLocation);

			MailboxWalk moonWalk = MailboxWalk.create(context, mbox, domainUid, server.value);

			moonWalk.folders((sc, allFolders) -> {
				monitor.begin(allFolders.size(),
						"[" + entry.email + "] Dealing with " + allFolders.size() + " folders");
				for (ListInfo f : allFolders) {
					String fn = f.getName();
					monitor.progress(1, "sync " + fn);
					if (!f.isSelectable() || fn.startsWith("Dossiers partagés/")
							|| fn.startsWith("Autres utilisateurs/")) {
						continue;
					}
					try {
						sc.select(fn);
						dedupByHeader(monitor, mbox, sc, MailApiHeaders.X_BM_INTERNAL_ID, fn);
						dedupByHeader(monitor, mbox, sc, "Message-ID", fn);
					} catch (IMAPException e) {
						monitor.log("Fail to select {} on mailbox {}", Level.WARN, fn, mbox.value.name);
					}

				}

			}, monitor);

			monitor.end();

		}

		private void dedupByHeader(RepairTaskMonitor monitor, ItemValue<Mailbox> mbox, StoreClient sc,
				String dedupByHeader, String fn) {
			String[] wanted = new String[] { dedupByHeader };

			InternalDate[] allMsgs = sc.uidFetchInternalDate("1:*");
			List<Integer> allUids = Arrays.stream(allMsgs).map(id -> id.getUid()).collect(Collectors.toList());
			Map<String, List<Integer>> matchingUids = duplicatesByHeaderValue(sc, dedupByHeader, wanted, allUids);
			int distinct = matchingUids.keySet().size();
			if (distinct > 0) {
				Iterator<Entry<String, List<Integer>>> it = matchingUids.entrySet().iterator();
				while (it.hasNext()) {
					Entry<String, List<Integer>> e = it.next();
					if (e.getValue().size() <= 1) {
						it.remove();
					}
				}
				// we only have duplicates in the map
				List<Integer> toDel = new ArrayList<>(matchingUids.size());
				for (Entry<String, List<Integer>> e : matchingUids.entrySet()) {
					List<Integer> listOfDups = e.getValue();
					Collections.sort(listOfDups);
					// only keep the biggest imapUid
					toDel.addAll(listOfDups.subList(0, listOfDups.size() - 1));
				}
				if (!toDel.isEmpty()) {
					monitor.notify("Purge {} dups / {} distinct {} with dups / {} total in {} on mbox {}", toDel.size(),
							distinct, dedupByHeader, allUids.size(), fn, mbox.value.name);
					FlagsList fl = new FlagsList();
					fl.add(Flag.SEEN);
					fl.add(Flag.DELETED);
					sc.uidStore(toDel, fl, true);
					sc.expunge();
				}
			}
		}

		private Map<String, List<Integer>> duplicatesByHeaderValue(StoreClient sc, String dedupByHeader,
				String[] wanted, List<Integer> allUids) {
			Map<String, List<Integer>> matchingUids = new HashMap<>();
			for (List<Integer> slice : Lists.partition(allUids, 5000)) {
				Collection<IMAPHeaders> headers = sc.uidFetchHeaders(slice, wanted);
				for (IMAPHeaders h : headers) {
					String iid = h.getRawHeader(dedupByHeader);
					if (!Strings.isNullOrEmpty(iid)) {
						List<Integer> uids = matchingUids.computeIfAbsent(iid, k -> new ArrayList<>(2));
						uids.add(h.getUid());
					}
				}
			}
			return matchingUids;
		}
	}

}
