/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2022
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.backend.mail.replica.service.internal;

import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import javax.sql.DataSource;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.backend.mail.api.IMailConversationInternalSync;
import net.bluemind.backend.mail.api.IMailboxFoldersByContainer;
import net.bluemind.backend.mail.replica.api.IMailReplicaUids;
import net.bluemind.backend.mail.replica.persistence.InternalConversation;
import net.bluemind.backend.mail.replica.persistence.InternalConversation.InternalMessageRef;
import net.bluemind.backend.mail.replica.persistence.MailboxRecordStore;
import net.bluemind.core.api.fault.ErrorCode;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.api.IContainers;
import net.bluemind.core.container.model.Container;
import net.bluemind.core.container.model.ContainerDescriptor;
import net.bluemind.core.container.model.Item;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.container.model.acl.Verb;
import net.bluemind.core.container.service.internal.RBACManager;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.core.task.api.TaskRef;
import net.bluemind.core.task.service.IServerTask;
import net.bluemind.core.task.service.IServerTaskMonitor;
import net.bluemind.core.task.service.ITasksManager;
import net.bluemind.index.mail.Sudo;
import net.bluemind.mailbox.api.IMailboxAclUids;
import net.bluemind.mailbox.api.Mailbox.Type;
import net.bluemind.user.api.IUser;
import net.bluemind.user.api.User;

public class MailConversationInternalSyncService implements IMailConversationInternalSync {

	private final RBACManager rbacManager;
	private final BmContext context;
	private final Container container;
	private final MailboxRecordStore recordStore;
	private final ConversationStoreService storeService;
	private static final Logger logger = LoggerFactory.getLogger(MailConversationInternalSyncService.class);

	public MailConversationInternalSyncService(BmContext context, DataSource ds, Container conversationContainer) {
		this.container = conversationContainer;
		this.storeService = new ConversationStoreService(ds, context.getSecurityContext(), conversationContainer);
		this.rbacManager = RBACManager.forContext(context)
				.forContainer(IMailboxAclUids.uidForMailbox(conversationContainer.owner));
		this.context = context;
		this.recordStore = new MailboxRecordStore(ds);
	}

	@Override
	public TaskRef sync(boolean dry) {
		rbacManager.check(Verb.Write.name());

		return context.provider().instance(ITasksManager.class).run(new IServerTask() {

			@Override
			public void run(IServerTaskMonitor monitor) throws Exception {
				syncConversations(monitor, dry);
			}
		});
	}

	private void syncConversations(IServerTaskMonitor monitor, boolean dry) {

		User user = context.provider().instance(IUser.class, container.domainUid).get(container.owner);

		try (Sudo su = new Sudo(user.login, container.domainUid)) {

			if (!dry) {
				monitor.log("Reset all conversations of user {}", container.owner);
				storeService.deleteAll();
			}

			ServerSideServiceProvider provider = ServerSideServiceProvider.getProvider(su.context);
			IMailboxFoldersByContainer folders = provider.instance(IMailboxFoldersByContainer.class,
					IMailReplicaUids.subtreeUid(container.domainUid, Type.user, container.owner));

			monitor.log("Synchronizing conversations of user {}", container.owner);
			Set<Long> folderIds = getFolderIds(monitor, folders);

			try {
				if (!dry) {
					monitor.log("Synchronizing conversations of user {} t_mailbox_record --> v_conversation_by_folder",
							container.owner);
					syncRecordsToVirtualLookupTable(monitor, folderIds);
				}
				monitor.log("Synchronizing conversations of user {} t_mailbox_record --> t_conversations",
						container.owner);
				syncRecordsToConversationsTable(monitor, dry, folderIds);
			} catch (Exception e) {
				logger.warn("Cannot resync conversations", e);
				monitor.end(false, e.getMessage(), null);
				return;
			}
		}
		monitor.end(true, "", "");
	}

	private void syncRecordsToVirtualLookupTable(IServerTaskMonitor monitor, Set<Long> folderIds) throws SQLException {
		for (Long folder : folderIds) {
			monitor.log("Starting resync of folder {}", folder);
			monitor.log("Deleting all virtual conversation entries of folder {}", folder);
			recordStore.deleteVirtualConversations(folder);
			Set<Long> conversationIds = recordStore.getRecordConversationIds(folder);
			monitor.log("Found a total of {} conversations in records of folder {}", conversationIds.size(), folder);
			for (Long conversation : conversationIds) {
				recordStore.triggerConversationUpdate(folder, conversation);
			}
		}
	}

	private void syncRecordsToConversationsTable(IServerTaskMonitor monitor, boolean dry, Set<Long> folderIds)
			throws SQLException {
		Set<Long> conversationIds = new HashSet<>(recordStore.getConversationIds(folderIds));
		monitor.log("Found a total of {} converations in {} folders", conversationIds.size(), folderIds.size());
		for (Long conversationId : conversationIds) {
			syncSingleConversation(monitor, conversationId, dry, container.owner);
		}
	}

	private void syncSingleConversation(IServerTaskMonitor monitor, Long conversationId, boolean dry, String owner)
			throws SQLException {
		List<InternalMessageRef> allItemsByConversation = recordStore.getAllItemsByConversation(conversationId, owner);
		monitor.log("Handling converation {} containing {} items", conversationId, allItemsByConversation.size());
		String conversationUid = Long.toHexString(conversationId);
		ItemValue<InternalConversation> conversation = storeService.get(conversationUid, null);
		boolean creation = conversation == null;
		if (creation) {
			conversation = ItemValue.create(Item.create(conversationUid, 0), new InternalConversation());
		}
		boolean updated = false;
		conversation.value.messageRefs = new ArrayList<>(conversation.value.messageRefs);

		for (InternalMessageRef recordItem : allItemsByConversation) {
			updated |= addRecordIfNecessary(conversation.value.messageRefs, recordItem);
		}
		if (updated) {
			saveConversation(monitor, conversationId, dry, conversation, creation);
		}
	}

	private void saveConversation(IServerTaskMonitor monitor, Long conversationId, boolean dry,
			ItemValue<InternalConversation> conversation, boolean creation) {
		monitor.log("Updates of conversation {} detected: {} total messages....", conversationId,
				conversation.value.messageRefs.size());
		if (!dry) {
			if (creation) {
				try {
					storeService.create(conversation.uid, createDisplayName(conversation.uid), conversation.value);
				} catch (ServerFault e) {
					if (e.getCode() == ErrorCode.ALREADY_EXISTS) {
						monitor.log("Conversation {} has been created in the meantime. Ignoring this entry",
								conversation.uid);
					} else {
						throw e;
					}
				}
			} else {
				storeService.update(conversation.uid, createDisplayName(conversation.uid), conversation.value);
			}
		}
	}

	private Set<Long> getFolderIds(IServerTaskMonitor monitor, IMailboxFoldersByContainer folders) {
		IContainers containers = context.getServiceProvider().instance(IContainers.class);
		return folders.all().stream().map(folder -> IMailReplicaUids.mboxRecords(folder.uid)).map(folder -> {
			ContainerDescriptor containerDescriptor = containers.get(folder);
			monitor.log("Found folder {}:{}", containerDescriptor.uid, containerDescriptor.name);
			return containerDescriptor.internalId;
		}).collect(Collectors.toSet());
	}

	private boolean addRecordIfNecessary(List<InternalMessageRef> messageRefs, InternalMessageRef recordItem) {
		boolean contains = messageRefs.stream()
				.anyMatch(ref -> ref.itemId == recordItem.itemId && ref.folderId == recordItem.folderId);
		if (!contains) {
			messageRefs.add(recordItem);
		}
		return !contains;
	}

	private String createDisplayName(String uid) {
		return "conversation_" + uid;
	}

}
