/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2022
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.backend.mail.replica.service.internal;

import java.util.Arrays;
import java.util.function.Supplier;

public class InternalIdHeader {

	public final String owner;
	public final String installationId;
	public final long internalId;

	private InternalIdHeader(String owner, String installationId, long internalId) {
		this.owner = owner;
		this.installationId = installationId;
		this.internalId = internalId;
	}

	public static InternalIdHeader parse(String value) {
		return getParser(value).get();
	}

	private static Supplier<InternalIdHeader> getParser(String value) {
		if (value.contains("#") && value.contains(":")) {
			return new NonFoldableFormatParser(value); // backwards compatibility with older releases, see BM-20267
		} else {
			return new FoldableFormatParser(value);
		}
	}

	private static class NonFoldableFormatParser implements Supplier<InternalIdHeader> {

		private final String value;

		public NonFoldableFormatParser(String value) {
			this.value = value;
		}

		@Override
		public InternalIdHeader get() {
			int instIdx = value.lastIndexOf(':');
			int ownerIdx = value.lastIndexOf('#');
			String owner = value.substring(0, ownerIdx);
			String instId = value.substring(ownerIdx + 1, instIdx);
			long internalId = Long.parseLong(value.substring(instIdx + 1));
			return new InternalIdHeader(owner, instId, internalId);
		}

	}

	private static class FoldableFormatParser implements Supplier<InternalIdHeader> {

		private final String value;

		public FoldableFormatParser(String value) {
			this.value = value;
		}

		@Override
		public InternalIdHeader get() {
			String[] split = value.split(" ");
			long internalId = Long.parseLong(split[split.length - 1]);
			String instId = split[split.length - 2];
			String owner = null;
			if (split.length == 3) {
				owner = split[0];
			} else { // owner may contain spaces
				String[] trimmed = Arrays.copyOf(split, split.length - 2);
				owner = String.join(" ", trimmed);
			}
			return new InternalIdHeader(owner, instId, internalId);
		}

	}

}
