/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2019
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.core.sessions.deletion.hz;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.core.sessions.ISessionDeletionListener;
import net.bluemind.domain.api.Domain;
import net.bluemind.domain.api.IDomains;
import net.bluemind.hornetq.client.MQ;
import net.bluemind.hornetq.client.OOPMessage;
import net.bluemind.hornetq.client.Producer;
import net.bluemind.hornetq.client.Topic;
import net.bluemind.user.api.IUser;
import net.bluemind.user.api.User;

public class PropagateSessionDeletions implements ISessionDeletionListener {

	private static final Logger logger = LoggerFactory.getLogger(PropagateSessionDeletions.class);

	private final CompletableFuture<Producer> producer;
	private final ServerSideServiceProvider provider = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM);

	public PropagateSessionDeletions() {
		producer = MQ.init().thenApply(v -> MQ.getProducer(Topic.CORE_SESSIONS));
	}

	@Override
	public void deleted(String identity, String sid, SecurityContext securityContext) {
		Producer prod = producer.getNow(null);
		if (prod != null) {
			OOPMessage cm = MQ.newMessage();
			String userUid = securityContext.getSubject();
			String domainUid = securityContext.getContainerUid();

			ItemValue<User> user = provider.instance(IUser.class, domainUid).getComplete(userUid);

			cm.putStringList("emails", getEmails(domainUid, user));
			cm.putStringProperty("external-id", user.externalId);
			cm.putStringProperty("sender", identity);
			cm.putStringProperty("operation", "logout");
			cm.putStringProperty("sid", sid);
			prod.send(cm);
			logger.debug("MQ: logout {} sent.", sid);
		} else {
			logger.warn("MQ is missing, logout support will fail");
		}
	}

	private List<String> getEmails(String domainUid, ItemValue<User> user) {
		try {
			Set<String> emails = new HashSet<>();
			ItemValue<Domain> domain = provider.instance(IDomains.class).get(domainUid);
			emails.add(user.value.login + "@" + domain.uid);
			user.value.emails.forEach(email -> {
				emails.add(email.address);
				if (email.allAliases) {
					domain.value.aliases.forEach(alias -> emails.add(email.localPart() + "@" + alias));
				}
			});
			return new ArrayList<>(emails);
		} catch (ServerFault e) {
			logger.warn("Cannot get user emails by {}@{}", user.uid, domainUid);
			return Collections.emptyList();
		}
	}

}
