/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2022
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.videoconferencing.utils;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.lang.reflect.Field;
import java.net.HttpURLConnection;
import java.net.InetSocketAddress;
import java.net.MalformedURLException;
import java.net.Proxy;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Strings;

import io.netty.handler.codec.http.HttpMethod;
import io.vertx.core.json.JsonObject;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.system.api.ISystemConfiguration;
import net.bluemind.system.api.SysConfKeys;

public class ApiHttpHelper {

	private static final Logger logger = LoggerFactory.getLogger(ApiHttpHelper.class);
	private final String bearerToken;

	private static final int TIMEOUT = 10000;

	public ApiHttpHelper(String bearerToken) {
		this.bearerToken = bearerToken;
	}

	private HttpURLConnection connect(String url) throws MalformedURLException, IOException {
		Map<String, String> sysConfMap = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM)
				.instance(ISystemConfiguration.class).getValues().values;
		String proxyEnabled = sysConfMap.get(SysConfKeys.http_proxy_enabled.name());
		if (proxyEnabled == null || proxyEnabled.trim().isEmpty() || !proxyEnabled.equals("true")) {
			return (HttpURLConnection) new URL(url).openConnection();
		} else {
			Proxy proxy = new Proxy(Proxy.Type.HTTP,
					new InetSocketAddress(sysConfMap.get(SysConfKeys.http_proxy_hostname.name()),
							Integer.valueOf(sysConfMap.get(SysConfKeys.http_proxy_port.name()))));
			return (HttpURLConnection) new URL(url).openConnection(proxy);
		}

	}

	public JsonObject execute(String spec, HttpMethod method, String body) {
		HttpURLConnection conn = null;
		try {
			conn = connect(spec);
			if (method.name().equals("PATCH")) {
				try {
					final Object target;
					if (conn instanceof sun.net.www.protocol.https.HttpsURLConnectionImpl) {
						final Field delegate = sun.net.www.protocol.https.HttpsURLConnectionImpl.class
								.getDeclaredField("delegate");
						delegate.setAccessible(true);
						target = delegate.get(conn);
					} else {
						target = conn;
					}
					final Field f = HttpURLConnection.class.getDeclaredField("method");
					f.setAccessible(true);
					f.set(target, "PATCH");
				} catch (IllegalAccessException | NoSuchFieldException ex) {
					throw ex;
				}
			} else {
				conn.setRequestMethod(method.name());
			}
			conn.setRequestProperty("Authorization", "Bearer " + bearerToken);
			conn.setRequestProperty("Accept", "application/json");
			conn.setRequestProperty("Content-Type", "application/json; utf-8");
			conn.setConnectTimeout(TIMEOUT);
			conn.setReadTimeout(TIMEOUT);

			if (body != null) {
				conn.setDoOutput(true);
				byte[] data = body.getBytes();
				try (OutputStream os = conn.getOutputStream()) {
					os.write(data, 0, data.length);
				}
			}

			conn.disconnect();

			int responseCode = conn.getResponseCode();
			if (responseCode >= 400) {
				String response = getResponse(conn.getErrorStream());
				throw new ServerFault(response);
			}

			String response = getResponse(conn.getInputStream());

			if (!Strings.isNullOrEmpty(response)) {
				return new JsonObject(response);
			}

			return new JsonObject();

		} catch (Exception e) {
			logger.warn("Exception while calling {}:{}", spec, method.name(), e);
			throw new ServerFault(e.getMessage());
		} finally {
			if (conn != null) {
				conn.disconnect();
			}
		}

	}

	private String getResponse(InputStream is) throws IOException, UnsupportedEncodingException {
		StringBuilder response = new StringBuilder();
		try (BufferedReader br = new BufferedReader(new InputStreamReader(is, StandardCharsets.UTF_8.name()))) {
			String responseLine = null;
			while ((responseLine = br.readLine()) != null) {
				response.append(responseLine.trim());
			}
		}
		return response.toString();
	}
}
