/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.backend.cyrus.bmgroups;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.google.common.base.Stopwatch;

import io.vertx.core.Handler;
import io.vertx.core.buffer.Buffer;
import io.vertx.core.net.NetSocket;
import net.bluemind.config.Token;
import net.bluemind.core.api.AsyncHandler;
import net.bluemind.core.api.fault.ErrorCode;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.caches.registry.CacheRegistry;
import net.bluemind.core.caches.registry.ICacheRegistration;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.rest.http.HttpClientProvider;
import net.bluemind.core.rest.http.ILocator;
import net.bluemind.core.rest.http.VertxPromiseServiceProvider;
import net.bluemind.domain.api.IDomainsPromise;
import net.bluemind.group.api.IGroupPromise;
import net.bluemind.network.topology.Topology;
import net.bluemind.user.api.IUserPromise;
import net.bluemind.user.api.User;

public class GroupProtocolHandler implements Handler<Buffer> {
	private static final Logger logger = LoggerFactory.getLogger(GroupProtocolHandler.class);
	private final NetSocket socket;
	private final Handler<Throwable> exceptionHandler;
	private final VertxPromiseServiceProvider provider;
	private static final Cache<String, ItemValue<User>> usersCache = Caffeine.newBuilder().recordStats()
			.expireAfterAccess(10, TimeUnit.MINUTES).build();
	private static final Cache<String, List<String>> memberOfCache = Caffeine.newBuilder().recordStats()
			.expireAfterWrite(10, TimeUnit.MINUTES).build();

	public static class CacheRegistration implements ICacheRegistration {
		@Override
		public void registerCaches(CacheRegistry cr) {
			cr.register("group-protocol-users", usersCache);
			cr.register("group-protocol-memberof", memberOfCache);
		}
	}

	public static Cache<String, ItemValue<User>> getUsersCache() {
		return usersCache;
	}

	public static Cache<String, List<String>> getMemberOfCache() {
		return memberOfCache;
	}

	private static class UserAndGroups {
		public UserAndGroups(ItemValue<User> user, List<String> groups) {
			this.user = user;
			this.groups = groups;
		}

		ItemValue<User> user;
		List<String> groups;
	}

	public GroupProtocolHandler(HttpClientProvider clientProvider, NetSocket socket) {
		this.socket = socket;
		ILocator cachingLocator = (String service, AsyncHandler<String[]> asyncHandler) -> {
			String core = Topology.get().core().value.address();
			String[] resp = new String[] { core };
			asyncHandler.success(resp);
		};
		this.exceptionHandler = (Throwable e) -> {
			logger.error("error: {}", e.getMessage(), e);
			socket.write(ko(String.format("error: %s", e.getMessage())));
			socket.close();
		};

		this.provider = new VertxPromiseServiceProvider(clientProvider, cachingLocator, Token.admin0());

		socket.exceptionHandler(this.exceptionHandler);
	}

	@Override
	public void handle(Buffer event) {
		if (logger.isDebugEnabled()) {
			logger.debug("C: {}byte(s)", event.length());
		}
		Stopwatch chrono = Stopwatch.createStarted();
		String loginAtDomain = event.toString().trim();
		loginAtDomain = loginAtDomain.replace('^', '.');
		logger.debug("search for login {}", loginAtDomain);
		if ("admin0".equals(loginAtDomain)) {
			socket.write(ok(null), ar -> {
				logger.debug("Dealt with admin0 in {}ms.", chrono.elapsed(TimeUnit.MILLISECONDS));
				socket.close();
			});
			return;
		}

		if (loginAtDomain.indexOf('@') == -1) {
			socket.write(ko(String.format("Invalid login: %s", loginAtDomain)), ar -> socket.close());
			logger.debug("Dealt with invalid login: '{}' in {}ms", loginAtDomain,
					chrono.elapsed(TimeUnit.MILLISECONDS));
			return;
		}
		String[] splitted = loginAtDomain.split("@");
		String login = splitted[0];
		String domain = splitted[1];

		if (login.startsWith("group:") && login.substring("group:".length()).equals(domain)) {
			domainAcl(chrono, login, domain);
		} else if (login.startsWith("group:")) {
			groupAcl(chrono, login, domain);
		} else {
			userAcl(chrono, login, domain);
		}

	}

	private void userAcl(Stopwatch chrono, String login, String domain) {
		String cacheKey = domain + "-" + login;
		ItemValue<User> userFromCache = usersCache.getIfPresent(cacheKey);
		CompletableFuture<ItemValue<User>> userFuture = null;
		if (userFromCache != null) {
			userFuture = CompletableFuture.completedFuture(userFromCache);
		} else {
			logger.debug("Miss for cacheKey {} ({}ms elapsed)", cacheKey, chrono.elapsed(TimeUnit.MILLISECONDS));
			userFuture = provider.instance(IUserPromise.class, domain).byLogin(login).thenCompose(user -> {
				if (user == null) {
					logger.debug("{} is not a login, going with uid lookup", login);
					return provider.instance(IUserPromise.class, domain).getComplete(login);
				} else {
					return CompletableFuture.completedFuture(user);
				}
			});
		}
		userFuture.thenCompose(user -> {
			if (user == null) {
				throw new ServerFault("user " + login + "@" + domain + " not found", ErrorCode.NOT_FOUND);
			}

			if (userFromCache == null) {
				usersCache.put(cacheKey, user);
				logger.debug("Populate userCache with {} ({}ms elapsed)", cacheKey,
						chrono.elapsed(TimeUnit.MILLISECONDS));
			}
			List<String> memberOfs = memberOfCache.getIfPresent(domain + "-" + user.uid);
			if (memberOfs != null) {
				return CompletableFuture.completedFuture(new UserAndGroups(user, memberOfs));
			} else {
				return provider.instance(IUserPromise.class, domain).memberOfGroups(user.uid)

						.thenApply(g -> {
							memberOfCache.put(domain + "-" + user.uid, g);
							return new UserAndGroups(user, g);
						});
			}
		}).thenAccept(groupsAndUser -> {
			logger.debug("time to find user and memberof time {}ms", chrono.elapsed(TimeUnit.MILLISECONDS));
			List<String> ret = new ArrayList<>(groupsAndUser.groups.size());
			for (String g : groupsAndUser.groups) {
				ret.add("group:" + g + "@" + domain);
			}
			logger.debug("found user {}@{}, memberof {}", login, domain, ret);

			String res = String.join(",", ret);
			if (!res.isEmpty()) {
				res += ",";
			}
			// default groups (dmail + useruid)
			res += "group:" + domain + "@" + domain + "," + groupsAndUser.user.uid + "@" + domain;
			socket.write(ok(res), ar -> {
				socket.close();
				long elapsed = chrono.elapsed(TimeUnit.MILLISECONDS);
				if (elapsed >= 20) {
					logger.info("userAcl for {} in {}ms.", login, elapsed);
				} else {
					logger.debug("userAcl for {} in {}ms.", login, elapsed);
				}
			});
		}).exceptionally(t -> {
			exceptionHandler.handle(t);
			return null;
		});
	}

	private void groupAcl(Stopwatch chrono, String login, String domain) {
		String uid = login.substring("group:".length());
		provider.instance(IGroupPromise.class, domain).getComplete(uid).thenAccept(value -> {
			if (value == null) {
				socket.write(ko(String.format("error: group %s not found", uid)), ar -> {
					logger.error("group resolution failed {} in {}ms", uid, chrono.elapsed(TimeUnit.MILLISECONDS));
					socket.close();
				});
				return;
			}

			socket.write(ok(null), ar -> {
				long elapsed = chrono.elapsed(TimeUnit.MILLISECONDS);
				if (elapsed >= 20) {
					logger.info("groupAcl for {} resolved in {}ms.", uid, elapsed);
				} else {
					logger.debug("groupAcl for {} resolved in {}ms.", uid, elapsed);
				}
				socket.close();
			});
		}).exceptionally(t -> {
			exceptionHandler.handle(t);
			return null;
		});
	}

	private void domainAcl(Stopwatch chrono, String login, String domain) {
		provider.instance(IDomainsPromise.class).get(domain).thenAccept(value -> {
			if (value == null) {
				socket.write(ko(String.format("error: group (domain) %s not found", domain)), ar -> {
					logger.error("{} does not match a valid domain in {}ms", login,
							chrono.elapsed(TimeUnit.MILLISECONDS));
					socket.close();
				});
				return;
			}

			socket.write(ok(null), ar -> {
				long elapsed = chrono.elapsed(TimeUnit.MILLISECONDS);
				if (elapsed >= 20) {
					logger.info("domainAcl for {} in {}ms.", login, elapsed);
				} else {
					logger.debug("domainAcl for {} in {}ms.", login, elapsed);
				}
				socket.close();
			});
		}).exceptionally(t -> {
			exceptionHandler.handle(t);
			return null;
		});
	}

	private Buffer ok(String msg) {
		logger.debug("ok {}", msg);
		Buffer message = Buffer.buffer();
		message.appendString("OK");
		if (msg != null) {
			message.appendString(msg);
		}
		Buffer ret = Buffer.buffer();
		ret.appendShort((short) message.length());
		ret.appendBuffer(message);
		logger.debug("return message {}", ret);
		return ret;
	}

	private Buffer ko(String msg) {
		logger.error("ko {}", msg);
		Buffer message = Buffer.buffer();
		message.appendString("KO ").appendString(msg);
		Buffer ret = Buffer.buffer().appendShort((short) message.length()).appendBuffer(message);
		logger.error("return message {}", ret);
		return ret;
	}

}
