/**
 * BEGIN LICENSE
 * Copyright Â© Blue Mind SAS, 2012-2019
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */

import axios from 'axios';

export default class {
    /**
     * Client implementation for net.bluemind.im.api.IInstantMessaging API.
     * @param {string} sid Authentication token
     * @param {string} base RPC base path
     * @constructor
     */
    constructor(sid, base = '') {
        this.sid = sid;
        this.base = base + "/api/im";
    }
     

    /**
     * @param {string} user1 
     * @param {string} user2 
     * @param {number} messagesCount 
     * @param {Object} canceller 
     * @return {Object}
     */
    getLastMessagesBetween(user1, user2, messagesCount, canceller) {
        let $url = this.base + "/_getLastMessagesBetween";

        const $params = {};
        if (user1 != null) {
            Object.assign($params, {user1});
        }
        if (user2 != null) {
            Object.assign($params, {user2});
        }
        if (messagesCount != null) {
            Object.assign($params, {messagesCount});
        }

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'POST',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {string} uid 
     * @param {Object} canceller 
     * @return {Object}
     */
    isActiveUser(uid, canceller) {
        let $url = this.base + "/_activeUser/{uid}";
        $url = $url.replace('{uid}', uid);

        const $params = {};

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'GET',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {string} sender 
     * @param {string} groupChatId 
     * @param {string[]} recipients 
     * @param {Object} canceller 
     * @return {Object}
     */
    sendGroupChatHistory(sender, groupChatId, recipients, canceller) {
        let $url = this.base + "/_sendGroupChatHistory/{groupChatId}";
        $url = $url.replace('{groupChatId}', groupChatId);

        const $params = {};
        if (sender != null) {
            Object.assign($params, {sender});
        }

        let $data, $responseType, $acceptHeader;

        if (recipients != null) {
            $data = recipients;
        }
        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'POST',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {string} jabberId 
     * @param {string} data 
     * @param {Object} canceller 
     * @return {Object}
     */
    setRoster(jabberId, data, canceller) {
        let $url = this.base + "/_setRoster";

        const $params = {};
        if (jabberId != null) {
            Object.assign($params, {jabberId});
        }

        let $data, $responseType, $acceptHeader;

        if (data != null) {
            $data = data;
        }
        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'POST',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {string} jabberId 
     * @param {Object} canceller 
     * @return {Object}
     */
    getRoster(jabberId, canceller) {
        let $url = this.base + "/_getRoster";

        const $params = {};
        if (jabberId != null) {
            Object.assign($params, {jabberId});
        }

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'GET',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    success(response) {
        if (response.data && response.data.errorType) {
            throw {response};
        }
        return response.data;
    }
    
    failure(error) {
        if (axios.isCancel(error)) {
            error = { message: "CANCELLED_BY_CLIENT" };
        }

        let fail;
        if (error.response && error.response.data && error.response.data.errorType) {
            fail = new Error(error.response.data.message);
            fail.name = error.response.data.errorType;
            fail.data = error.response.data;
        } else if (error.response && error.response.message) {
            fail = new Error(error.response.message);
            fail.name = error.response.status;
            fail.data = error.response;
        } else {
            fail = new Error(error.message);
            fail.data = error;
        }
        throw fail;
    }
}
