/**
 * BEGIN LICENSE
 * Copyright Â© Blue Mind SAS, 2012-2019
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */

import axios from 'axios';

export default class {
    /**
     * Client implementation for net.bluemind.hsm.api.IHSM API.
     * @param {string} sid Authentication token
     * @param {string} domainUid
     * @param {string} base RPC base path
     * @constructor
     */
    constructor(sid, domainUid , base = '') {
        this.sid = sid;
        this.base = base + "/api/hsm/{domainUid}";
        this.domainUid = domainUid;
        this.base = this.base.replace('{domainUid}',domainUid);
    }
     

    /**
     * @param {Object[]} promote 
     * @param {Object} canceller 
     * @return {Object}
     */
    promoteMultiple(promote, canceller) {
        let $url = this.base + "/_massPromote";

        const $params = {};

        let $data, $responseType, $acceptHeader;

        if (promote != null) {
            $data = promote;
        }
        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'POST',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {string} mailboxUid 
     * @param {Object} canceller 
     * @return {Object}
     */
    getSize(mailboxUid, canceller) {
        let $url = this.base + "/_getSize/{mailboxUid}";
        $url = $url.replace('{mailboxUid}', mailboxUid);

        const $params = {};

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'GET',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {string} mailboxUid 
     * @param {string} hsmId 
     * @param {Object} canceller 
     * @return {Object}
     */
    fetch(mailboxUid, hsmId, canceller) {
        let $url = this.base + "/_fetch/{mailboxUid}/{hsmId}";
        $url = $url.replace('{mailboxUid}', mailboxUid);
        $url = $url.replace('{hsmId}', hsmId);

        const $params = {};

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'GET',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {string} sourceMailboxUid 
     * @param {string} destMailboxUid 
     * @param {string[]} hsmIds 
     * @param {Object} canceller 
     * @return {Object}
     */
    copy(sourceMailboxUid, destMailboxUid, hsmIds, canceller) {
        let $url = this.base + "/_copy/{sourceMailboxUid}/{destMailboxUid}";
        $url = $url.replace('{sourceMailboxUid}', sourceMailboxUid);
        $url = $url.replace('{destMailboxUid}', destMailboxUid);

        const $params = {};

        let $data, $responseType, $acceptHeader;

        if (hsmIds != null) {
            $data = hsmIds;
        }
        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'POST',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    success(response) {
        if (response.data && response.data.errorType) {
            throw {response};
        }
        return response.data;
    }
    
    failure(error) {
        if (axios.isCancel(error)) {
            error = { message: "CANCELLED_BY_CLIENT" };
        }

        let fail;
        if (error.response && error.response.data && error.response.data.errorType) {
            fail = new Error(error.response.data.message);
            fail.name = error.response.data.errorType;
            fail.data = error.response.data;
        } else if (error.response && error.response.message) {
            fail = new Error(error.response.message);
            fail.name = error.response.status;
            fail.data = error.response;
        } else {
            fail = new Error(error.message);
            fail.data = error;
        }
        throw fail;
    }
}
