/**
 * BEGIN LICENSE
 * Copyright Â© Blue Mind SAS, 2012-2019
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */

import axios from 'axios';

export default class {
    /**
     * Client implementation for net.bluemind.calendar.api.IPublishCalendar API.
     * @param {string} sid Authentication token
     * @param {string} containerUid
     * @param {string} base RPC base path
     * @constructor
     */
    constructor(sid, containerUid , base = '') {
        this.sid = sid;
        this.base = base + "/api/calendars/publish/{containerUid}";
        this.containerUid = containerUid;
        this.base = this.base.replace('{containerUid}',containerUid);
    }
     

    /**
     * @param {Object} mode 
     * @param {Object} canceller 
     * @return {Object}
     */
    generateUrl(mode, canceller) {
        let $url = this.base + "/_generate/{mode}";
        $url = $url.replace('{mode}', mode);

        const $params = {};

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'PUT',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {string} url 
     * @param {Object} canceller 
     * @return {Object}
     */
    disableUrl(url, canceller) {
        let $url = this.base + "/_disable";

        const $params = {};

        let $data, $responseType, $acceptHeader;

        if (url != null) {
            $data = url;
        }
        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'POST',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {Object} mode 
     * @param {Object} canceller 
     * @return {Object}
     */
    getGeneratedUrls(mode, canceller) {
        let $url = this.base + "/generated/{mode}";
        $url = $url.replace('{mode}', mode);

        const $params = {};

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'GET',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {string} token 
     * @param {Object} canceller 
     * @return {Object}
     */
    publish(token, canceller) {
        let $url = this.base + "/{token}";
        $url = $url.replace('{token}', token);

        const $params = {};

        let $data, $responseType, $acceptHeader;

        $responseType = "blob";
        $acceptHeader = "*/*";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'GET',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {Object} mode 
     * @param {string} token 
     * @param {Object} canceller 
     * @return {Object}
     */
    createUrl(mode, token, canceller) {
        let $url = this.base + "/_create/{mode}";
        $url = $url.replace('{mode}', mode);

        const $params = {};
        if (token != null) {
            Object.assign($params, {token});
        }

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'PUT',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    success(response) {
        if (response.data && response.data.errorType) {
            throw {response};
        }
        return response.data;
    }
    
    failure(error) {
        if (axios.isCancel(error)) {
            error = { message: "CANCELLED_BY_CLIENT" };
        }

        let fail;
        if (error.response && error.response.data && error.response.data.errorType) {
            fail = new Error(error.response.data.message);
            fail.name = error.response.data.errorType;
            fail.data = error.response.data;
        } else if (error.response && error.response.message) {
            fail = new Error(error.response.message);
            fail.name = error.response.status;
            fail.data = error.response;
        } else {
            fail = new Error(error.message);
            fail.data = error;
        }
        throw fail;
    }
}
