/**
 * BEGIN LICENSE
 * Copyright Â© Blue Mind SAS, 2012-2019
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */

import axios from 'axios';

export default class {
    /**
     * Client implementation for net.bluemind.backend.mail.api.IMailConversationActions API.
     * @param {string} sid Authentication token
     * @param {string} conversationContainer
     * @param {string} replicatedMailboxUid
     * @param {string} base RPC base path
     * @constructor
     */
    constructor(sid, conversationContainer , replicatedMailboxUid , base = '') {
        this.sid = sid;
        this.base = base + "/api/mail_conversation/{conversationContainer}/{replicatedMailboxUid}";
        this.conversationContainer = conversationContainer;
        this.base = this.base.replace('{conversationContainer}',conversationContainer);
        this.replicatedMailboxUid = replicatedMailboxUid;
        this.base = this.base.replace('{replicatedMailboxUid}',replicatedMailboxUid);
    }
     

    /**
     * @param {number} folderDestinationId 
     * @param {Object} mailboxItems 
     * @param {Object} canceller 
     * @return {Object}
     */
    importItems(folderDestinationId, mailboxItems, canceller) {
        let $url = this.base + "/importItems/{folderDestinationId}";
        $url = $url.replace('{folderDestinationId}', folderDestinationId);

        const $params = {};

        let $data, $responseType, $acceptHeader;

        if (mailboxItems != null) {
            $data = mailboxItems;
        }
        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'PUT',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {string} targetMailboxUid 
     * @param {string[]} conversationUids 
     * @param {Object} canceller 
     * @return {Object}
     */
    move(targetMailboxUid, conversationUids, canceller) {
        let $url = this.base + "/move/{targetMailboxUid}";
        $url = $url.replace('{targetMailboxUid}', targetMailboxUid);

        const $params = {};

        let $data, $responseType, $acceptHeader;

        if (conversationUids != null) {
            $data = conversationUids;
        }
        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'POST',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {Object} flagUpdate 
     * @param {Object} canceller 
     * @return {Object}
     */
    addFlag(flagUpdate, canceller) {
        let $url = this.base + "/_addFlag";

        const $params = {};

        let $data, $responseType, $acceptHeader;

        if (flagUpdate != null) {
            $data = flagUpdate;
        }
        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'PUT',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {string[]} conversationUids 
     * @param {Object} canceller 
     * @return {Object}
     */
    multipleDeleteById(conversationUids, canceller) {
        let $url = this.base + "/_multipleDelete";

        const $params = {};

        let $data, $responseType, $acceptHeader;

        if (conversationUids != null) {
            $data = conversationUids;
        }
        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'POST',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {Object} flagUpdate 
     * @param {Object} canceller 
     * @return {Object}
     */
    deleteFlag(flagUpdate, canceller) {
        let $url = this.base + "/_deleteFlag";

        const $params = {};

        let $data, $responseType, $acceptHeader;

        if (flagUpdate != null) {
            $data = flagUpdate;
        }
        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'PUT',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {string} targetMailboxUid 
     * @param {string[]} conversationUids 
     * @param {Object} canceller 
     * @return {Object}
     */
    copy(targetMailboxUid, conversationUids, canceller) {
        let $url = this.base + "/copy/{targetMailboxUid}";
        $url = $url.replace('{targetMailboxUid}', targetMailboxUid);

        const $params = {};

        let $data, $responseType, $acceptHeader;

        if (conversationUids != null) {
            $data = conversationUids;
        }
        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'POST',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    success(response) {
        if (response.data && response.data.errorType) {
            throw {response};
        }
        return response.data;
    }
    
    failure(error) {
        if (axios.isCancel(error)) {
            error = { message: "CANCELLED_BY_CLIENT" };
        }

        let fail;
        if (error.response && error.response.data && error.response.data.errorType) {
            fail = new Error(error.response.data.message);
            fail.name = error.response.data.errorType;
            fail.data = error.response.data;
        } else if (error.response && error.response.message) {
            fail = new Error(error.response.message);
            fail.name = error.response.status;
            fail.data = error.response;
        } else {
            fail = new Error(error.message);
            fail.data = error;
        }
        throw fail;
    }
}
