/**
 * BEGIN LICENSE
 * Copyright Â© Blue Mind SAS, 2012-2019
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */

import axios from 'axios';

export default class {
    /**
     * Client implementation for net.bluemind.addressbook.api.IAddressBooksMgmt API.
     * @param {string} sid Authentication token
     * @param {string} base RPC base path
     * @constructor
     */
    constructor(sid, base = '') {
        this.sid = sid;
        this.base = base + "/api/mgmt/addressbooks";
    }
     

    /**
     * @param {string} containerUid 
     * @param {number} since 
     * @param {Object} canceller 
     * @return {Object}
     */
    backup(containerUid, since, canceller) {
        let $url = this.base + "/{containerUid}/_backupstream";
        $url = $url.replace('{containerUid}', containerUid);

        const $params = {};
        if (since != null) {
            Object.assign($params, {since});
        }

        let $data, $responseType, $acceptHeader;

        $responseType = "blob";
        $acceptHeader = "*/*";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'GET',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {string} domain 
     * @param {Object} canceller 
     * @return {Object}
     */
    reindexDomain(domain, canceller) {
        let $url = this.base + "/_reindexDomain";

        const $params = {};
        if (domain != null) {
            Object.assign($params, {domain});
        }

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'POST',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {string} containerUid 
     * @param {Object} canceller 
     * @return {Object}
     */
    reindex(containerUid, canceller) {
        let $url = this.base + "/{containerUid}/_reindex";
        $url = $url.replace('{containerUid}', containerUid);

        const $params = {};

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'POST',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {string} containerUid 
     * @param {stream} restoreStream 
     * @param {boolean} reset 
     * @param {Object} canceller 
     * @param {Object} onUploadProgress 
     * @return {Object}
     */
    restore(containerUid, restoreStream, reset, canceller, onUploadProgress) {
        let $url = this.base + "/{containerUid}/_restorestream";
        $url = $url.replace('{containerUid}', containerUid);

        const $params = {};
        if (reset != null) {
            Object.assign($params, {reset});
        }

        let $data, $responseType, $acceptHeader;

        if (restoreStream != null) {
            $data = restoreStream;
        }
        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'POST',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            }),
            onUploadProgress
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {string} containerUid 
     * @param {Object} descriptor 
     * @param {Object} canceller 
     * @return {Object}
     */
    update(containerUid, descriptor, canceller) {
        let $url = this.base + "/{containerUid}";
        $url = $url.replace('{containerUid}', containerUid);

        const $params = {};

        let $data, $responseType, $acceptHeader;

        if (descriptor != null) {
            $data = descriptor;
        }
        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'POST',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {Object} canceller 
     * @return {Object}
     */
    reindexAll(canceller) {
        let $url = this.base + "/_reindex";

        const $params = {};

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'POST',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {string} containerUid 
     * @param {Object} canceller 
     * @return {Object}
     */
    remove(containerUid, canceller) {
        let $url = this.base + "/{containerUid}";
        $url = $url.replace('{containerUid}', containerUid);

        const $params = {};

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'DELETE',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {string} containerUid 
     * @param {Object} canceller 
     * @return {Object}
     */
    getComplete(containerUid, canceller) {
        let $url = this.base + "/{containerUid}";
        $url = $url.replace('{containerUid}', containerUid);

        const $params = {};

        let $data, $responseType, $acceptHeader;

        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'GET',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    /**
     * @param {string} containerUid 
     * @param {Object} descriptor 
     * @param {boolean} isDefault 
     * @param {Object} canceller 
     * @return {Object}
     */
    create(containerUid, descriptor, isDefault, canceller) {
        let $url = this.base + "/{containerUid}";
        $url = $url.replace('{containerUid}', containerUid);

        const $params = {};
        if (isDefault != null) {
            Object.assign($params, {isDefault});
        }

        let $data, $responseType, $acceptHeader;

        if (descriptor != null) {
            $data = descriptor;
        }
        $responseType = "json";
        $acceptHeader = "application/json";
        let $headers = {
            'Accept' : $acceptHeader
        };

        if (this.sid) {
            $headers['X-BM-ApiKey'] = this.sid;
        }

        return axios({
            method: 'PUT',
            params: $params,
            url: $url,
            data: $data,
            responseType: $responseType,
            headers: $headers,
            cancelToken: new axios.CancelToken(function executor(c) {
                if (canceller) {
                    canceller.cancel = c;
                }
            })
        }).then(this.success).catch(this.failure);
    }

    success(response) {
        if (response.data && response.data.errorType) {
            throw {response};
        }
        return response.data;
    }
    
    failure(error) {
        if (axios.isCancel(error)) {
            error = { message: "CANCELLED_BY_CLIENT" };
        }

        let fail;
        if (error.response && error.response.data && error.response.data.errorType) {
            fail = new Error(error.response.data.message);
            fail.name = error.response.data.errorType;
            fail.data = error.response.data;
        } else if (error.response && error.response.message) {
            fail = new Error(error.response.message);
            fail.name = error.response.status;
            fail.data = error.response;
        } else {
            fail = new Error(error.message);
            fail.data = error;
        }
        throw fail;
    }
}
