import { messageUtils } from "@bluemind/mail";

const { MessageForwardAttributeSeparator, MessageReplyAttributeSeparator, MessageQuoteClasses, MessageQuoteOutlookId } =
    messageUtils;

const NOT_FOUND = "NOT_FOUND";
const THREE_DOTS_BUTTON_ID = "bm-three-dots-button";

export default {
    NOT_FOUND,
    THREE_DOTS_BUTTON_ID,

    findQuoteNodes(message, htmlDoc) {
        return (
            findFromNodeAndNextSiblings(htmlDoc, message) ||
            findLastTopLevelBlockquote(htmlDoc) ||
            findReplyOrForwardContentNodesNotInsideBlockquote(htmlDoc)
        );
    },

    /**
     * @returns the nodes in 'html' containing 'quote' and wrapping elements ('blockquote' tag, 'has wrote' header...).
     */
    findQuoteNodesUsingTextComparison(html, quote) {
        const rootDocument = new DOMParser().parseFromString(html, "text/html");
        const quoteElement = new DOMParser().parseFromString(quote, "text/html").body;
        const nodeWithQuoteInRootNode = findNodeContainingTextNodes(rootDocument, quoteElement);
        if (nodeWithQuoteInRootNode) {
            const nodeIterator = rootDocument.createNodeIterator(rootDocument.body, NodeFilter.SHOW_ALL, node =>
                node.isEqualNode(nodeWithQuoteInRootNode) ? NodeFilter.FILTER_ACCEPT : NodeFilter.FILTER_REJECT
            );
            const quoteNode = nodeIterator.nextNode(); // assumes the first one is the only one needed
            const quoteNodes = addQuoteHeaderAndWrappingElements(quoteNode);
            return quoteNodes;
        }
    },

    removeQuotes(rootNode, quoteNodes) {
        if (quoteNodes) {
            const nodeIterator = rootNode.createNodeIterator(rootNode, NodeFilter.SHOW_ALL, node =>
                quoteNodes.some(qn => node.isSameNode(qn)) ? NodeFilter.FILTER_ACCEPT : NodeFilter.FILTER_REJECT
            );

            const button = document.createElement("button");
            button.style.cssText += "border: 0; cursor: pointer; font-size: large";
            button.setAttribute("type", "button");
            button.innerHTML = "...";
            button.id = THREE_DOTS_BUTTON_ID;

            let node;
            let threeDotsButtonInserted = false;
            while ((node = nodeIterator.nextNode())) {
                if (threeDotsButtonInserted) {
                    node.parentNode.removeChild(node);
                } else {
                    node.parentNode.replaceChild(button, node);
                    threeDotsButtonInserted = true;
                }
            }
        }
        return rootNode;
    }
};

const SEP_IDS = [MessageReplyAttributeSeparator, MessageForwardAttributeSeparator, MessageQuoteOutlookId];
const SEP_CLASSES = MessageQuoteClasses;

function findReplyOrForwardContentNodesNotInsideBlockquote(htmlDoc) {
    // xpath example: //*[(@id="data-bm-reply-separator" or @id="data-bm-forward-separator" or contains(class,"data-bm-reply-separator") or contains(class,"data-bm-forward-separator")) and not(ancestor::blockquote)]
    const sepIdXpath = SEP_IDS.length ? SEP_IDS.map(sid => '@id="' + sid + '"').join(" or ") : "";
    const sepClassXpath = SEP_CLASSES.length
        ? " or " + SEP_CLASSES.map(sc => 'contains(class,"' + sc + '")').join(" or ")
        : "";
    const xpath = `//*[(${sepIdXpath}${sepClassXpath}) and not(ancestor::blockquote)]`;
    const xpathResult = htmlDoc?.evaluate(xpath, htmlDoc.body, null, XPathResult.ORDERED_NODE_ITERATOR_TYPE, null);
    let res = xpathResult?.iterateNext();
    const nodes = res ? [res] : undefined;
    while (res) {
        res = xpathResult?.iterateNext();
        if (res) {
            nodes.push(res);
        }
    }
    return nodes;
}

/** Find 'from' contact node (like "Georges Abitbol <g.abitol@lca.net>") and its following siblings. */
function findFromNodeAndNextSiblings(htmlDoc, message) {
    const toRegex =
        "(?:" +
        message.to.reduce((all, current) => {
            const allStr = all ? `${all}|` : "";
            const currentStr = current.dn ? `${current.dn}\\s*<${current.address}>` : current.address;
            return `${allStr}${currentStr}`;
        }, "") +
        ")";
    const matches = htmlDoc.body.innerText.match(new RegExp(toRegex));
    const matchingTo = matches && matches[0];
    if (matchingTo) {
        const to = message.to.find(
            ({ dn, address }) => (!dn || matchingTo.includes(dn)) && matchingTo.includes(address)
        );
        const toXPath = to.dn
            ? `contains(text(), "${to.dn}") and contains(text(), "${to.address}")`
            : `contains(text(), "${to.address}")`;
        const xpath = `//*[${toXPath} and not(ancestor::blockquote)]`;
        const fromNode = htmlDoc?.evaluate(
            xpath,
            htmlDoc.body,
            null,
            XPathResult.FIRST_ORDERED_NODE_TYPE,
            null
        )?.singleNodeValue;
        let nodes;
        if (fromNode) {
            const node = getHighestAncestorWithSameContent(fromNode);
            nodes = [node];
            const isOnlyChild = node.parentNode.children.length === 1;
            let sibling = isOnlyChild ? node.parentElement : node;
            while ((sibling = sibling.nextElementSibling)) {
                nodes.push(sibling);
            }
        }
        return nodes;
    }
}

function getHighestAncestorWithSameContent(node) {
    let currentNode = node;
    while (currentNode && currentNode.parentElement.textContent.trim() === node.textContent.trim()) {
        currentNode = currentNode.parentElement;
    }
    return currentNode;
}

function addQuoteHeaderAndWrappingElements(quoteNode) {
    let blockquoteNode = quoteNode.tagName === "BLOCKQUOTE" ? quoteNode : undefined;
    let currentNode = quoteNode;
    while (!blockquoteNode && currentNode) {
        if (currentNode.parentElement && currentNode.parentElement.tagName === "BLOCKQUOTE") {
            blockquoteNode = currentNode.parentNode;
        }
        currentNode = currentNode.parentNode;
    }
    return blockquoteNode ? [findQuoteHeaderFromBlockquote(blockquoteNode), blockquoteNode] : undefined;
}

function findQuoteHeaderFromBlockquote(blockquoteNode) {
    let quoteHeaderNode;
    let node = blockquoteNode.previousSibling;
    while (!quoteHeaderNode && node) {
        if (node.textContent && node.textContent.replace(/\s+/, "").length) {
            quoteHeaderNode = node;
        }
        node = node.previousSibling;
    }
    return quoteHeaderNode;
}

function findNodeContainingTextNodes(rootDocument, nodeWithTextNodesToMatch) {
    const spacesRegex = /\s|\n|&nbsp;/g;
    const textToMatch = nodeWithTextNodesToMatch.innerText.replace(spacesRegex, "");

    let deepestMatchingNode;
    let deepestMatchingNodeDepth = 0;
    const nodeIterator = rootDocument.createNodeIterator(
        rootDocument.body,
        NodeFilter.SHOW_ALL,
        () => NodeFilter.FILTER_ACCEPT
    );
    let node;
    while ((node = nodeIterator.nextNode())) {
        node.depth = node?.parentNode.depth + 1 || 0;
        if (
            node.innerText &&
            node.innerText.replace(spacesRegex, "").includes(textToMatch) &&
            node.depth > deepestMatchingNodeDepth
        ) {
            deepestMatchingNode = node;
        }
    }
    return deepestMatchingNode;
}

/** Find the last top level Blockquote. */
function findLastTopLevelBlockquote(htmlDoc) {
    const xpath = `(//blockquote[not(ancestor::blockquote)])[last()]`;
    const node = htmlDoc?.evaluate(
        xpath,
        htmlDoc.body,
        null,
        XPathResult.FIRST_ORDERED_NODE_TYPE,
        null
    )?.singleNodeValue;
    return node && [node];
}
