import { inject } from "@bluemind/inject";
import { loadingStatusUtils } from "@bluemind/mail";
import EventHelper from "./helpers/EventHelper";
import { FETCH_EVENT, SET_EVENT_STATUS, ACCEPT_COUNTER_EVENT, DECLINE_COUNTER_EVENT } from "~/actions";
import {
    RESET_ACTIVE_MESSAGE,
    SET_BLOCK_REMOTE_IMAGES,
    SET_CURRENT_EVENT,
    SET_CURRENT_EVENT_STATUS
} from "~/mutations";
import { Verb } from "@bluemind/core.container.api";

const { LoadingStatus } = loadingStatusUtils;

export default {
    state: {
        currentEvent: { loading: LoadingStatus.NOT_LOADED },
        remoteImages: { mustBeBlocked: false }
    },
    actions: {
        async [FETCH_EVENT]({ commit }, { message, mailbox }) {
            commit(SET_CURRENT_EVENT, { loading: LoadingStatus.LOADING });
            try {
                const mailboxOwner = message.eventInfo.isResourceBooking
                    ? message.eventInfo.resourceUid
                    : mailbox.owner;
                const calendar = getCalendarUid(mailboxOwner, message.eventInfo.isResourceBooking);
                const isWritable = await isCalendarWritable(calendar);
                const events = await inject("CalendarPersistence", calendar).getByIcsUid(message.eventInfo.icsUid);
                let event = findEvent(events, message.eventInfo.recuridIsoDate);

                event = EventHelper.adapt(
                    event,
                    mailboxOwner,
                    message.from.address,
                    message.eventInfo.recuridIsoDate,
                    calendar,
                    isWritable
                );
                commit(SET_CURRENT_EVENT, event);
            } catch {
                commit(SET_CURRENT_EVENT, { loading: LoadingStatus.ERROR });
                throw "Event not found";
            }
        },

        async [SET_EVENT_STATUS]({ state, commit }, { status }) {
            const previous = { ...state.currentEvent };
            try {
                commit(SET_CURRENT_EVENT_STATUS, { status });
                await inject("CalendarPersistence", previous.calendar).update(
                    state.currentEvent.uid,
                    state.currentEvent.serverEvent.value,
                    true
                );
            } catch {
                commit(SET_CURRENT_EVENT_STATUS, { status: previous.status });
            }
        },

        [ACCEPT_COUNTER_EVENT]({ state, commit }) {
            return updateCounterEvent({ state, commit }, EventHelper.applyCounter);
        },

        [DECLINE_COUNTER_EVENT]({ state, commit }) {
            return updateCounterEvent({ state, commit }, EventHelper.removeCounter);
        }
    },
    mutations: {
        [SET_CURRENT_EVENT](state, event) {
            state.currentEvent = event;
        },
        [SET_CURRENT_EVENT_STATUS](state, { status }) {
            state.currentEvent.status = status;
            EventHelper.setStatus(state.currentEvent, status);
        },
        [SET_BLOCK_REMOTE_IMAGES](state, mustBeBlocked) {
            state.remoteImages.mustBeBlocked = mustBeBlocked;
        },
        [RESET_ACTIVE_MESSAGE](state) {
            state.currentEvent = { loading: LoadingStatus.NOT_LOADED };
        }
    }
};

async function updateCounterEvent({ state, commit }, updateFunction) {
    let updatedEvent = JSON.parse(JSON.stringify(state.currentEvent));

    const recuridIsoDate = state.currentEvent.counter.occurrence
        ? state.currentEvent.counter.occurrence.recurid.iso8601
        : undefined;
    updateFunction(updatedEvent, state.currentEvent.counter.originator, recuridIsoDate);

    updatedEvent = EventHelper.adapt(
        updatedEvent.serverEvent,
        state.currentEvent.mailboxOwner,
        state.currentEvent.counter.originator
    );

    commit(SET_CURRENT_EVENT, updatedEvent);
    await inject("CalendarPersistence").update(state.currentEvent.uid, updatedEvent.serverEvent.value, true);
}

const findEvent = (events, recuridIsoDate) => {
    return events.find(
        event =>
            !recuridIsoDate || event.value.occurrences.some(occurrence => occurrence.recurid.iso8601 === recuridIsoDate)
    );
};

function getCalendarUid(owner, isRessource) {
    return isRessource ? `calendar:${owner}` : `calendar:Default:${owner}`;
}

async function isCalendarWritable(calendarUid) {
    const { userId } = inject("UserSession");
    if (calendarUid === getCalendarUid(userId)) {
        return [Verb.Read, Verb.Write];
    }
    return inject("ContainerManagementPersistence", calendarUid).canAccess([Verb.Write]);
}
