import Vue from "vue";

import { showNotification } from "@bluemind/commons/utils/notification";
import { extensions } from "@bluemind/extensions";
import router from "@bluemind/router";
import BmRoles from "@bluemind/roles";
import WebsocketClient from "@bluemind/sockjs";

import MailAlertRenderer from "./components/MailAlertRenderer";
import * as MailAlertComponents from "./components/MailAlerts";
import * as ThreadAlertComponents from "./components/MailThread/Alerts";
import MailApp from "./components/MailApp";
import mailRoutes from "./router";
import registerAPIClients from "./registerApiClients";

registerAPIClients();
router.addRoutes(mailRoutes);
registerMailtoHandler();
addNewMailNotification(window.bmcSessionInfos);
Vue.component("MailWebapp", MailApp);
Vue.component("MailAlertRenderer", MailAlertRenderer);
const AlertComponents = { ...MailAlertComponents, ...ThreadAlertComponents };
for (let component in AlertComponents) {
    Vue.component(component, AlertComponents[component]);
}

(async () => {
    if ("serviceWorker" in navigator) {
        try {
            const registration = await navigator.serviceWorker.register("service-worker.js");
            // eslint-disable-next-line no-console
            console.log("Registration succeeded. Scope is " + registration.scope);
        } catch (error) {
            // eslint-disable-next-line no-console
            console.log("Registration failed with " + error);
        }

        navigator.serviceWorker.addEventListener("message", event => {
            if (event.data.type === "ERROR") {
                // eslint-disable-next-line no-console
                console.warn(event.data.payload.message);
            }
        });

        navigator.serviceWorker.addEventListener("waiting", () => {
            // eslint-disable-next-line no-console
            console.warn(
                "A new service worker has installed, but it can't activate until all tabs running the current version have fully unloaded."
            );
        });

        navigator.serviceWorker.addEventListener("installed", event => {
            if (event.isUpdate) {
                // eslint-disable-next-line no-console
                console.warn("A new version of the site is available, please refresh the page.");
            }
        });
    }
})();

function registerMailtoHandler() {
    // Firefox based browsers registering popup has no "block" option, they will spam the user indefinitely
    if (window.navigator.registerProtocolHandler && !/firefox/i.test(navigator.userAgent)) {
        window.navigator.registerProtocolHandler("mailto", "/webapp/mail/%s", "Mailto Handler");
    }
}

async function addNewMailNotification(userSession) {
    if (userSession.roles.includes(BmRoles.HAS_MAIL)) {
        const application = extensions
            .get("net.bluemind.webapp", "application")
            .find(application => application.$id === "net.bluemind.webapp.mail.js");
        const icon = URL.createObjectURL(new Blob([application.icon.svg.toString()], { type: "image/svg+xml" }));
        const address = `${userSession.userId}.notifications.mails`;

        new WebsocketClient().register(address, async ({ data: { body: data } }) => {
            const tag = `notifications.mails`;
            const renotify = true;
            showNotification(data.title, { body: data.body, icon, badge: icon, image: icon, data, tag, renotify });
        });
    }
}
