/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2023
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.cli.index;

import java.util.List;
import java.util.Optional;
import java.util.stream.Stream;

import net.bluemind.cli.cmd.api.CliContext;
import net.bluemind.cli.cmd.api.ICmdLet;
import net.bluemind.cli.cmd.api.ICmdLetRegistration;
import net.bluemind.cli.utils.CliUtils;
import net.bluemind.directory.api.IDirectory;
import net.bluemind.lib.elasticsearch.ESearchActivator;
import net.bluemind.mailbox.api.IMailboxMgmt;
import net.bluemind.mailbox.api.SimpleShardStats;
import picocli.CommandLine.Command;
import picocli.CommandLine.Option;

@Command(name = "prune-mailspools", description = "Clean given indices:\n"
		+ " - delete indices having alias to unknown mailboxes,\n" //
		+ " - reset indices having no mailbox aliases,\n" //
		+ " - keep indices having active mailbox aliases.")
public class PruneMailspoolCommand implements ICmdLet, Runnable {

	@Option(names = "--indexes", description = "comma separated list of indexes (e.q. 2,5,6)", required = true)
	public String mailspoolNumbers;

	@Option(names = "--apply", description = "non-dry mode")
	public boolean apply = false;

	private CliContext ctx;

	private CliUtils cliUtils;

	@Override
	public void run() {
		List<String> domainUids = cliUtils.getDomainUids();
		IMailboxMgmt mboxMgmt = ctx.longRequestTimeoutAdminApi().instance(IMailboxMgmt.class, "global.virt");
		Stream.of(mailspoolNumbers.split(",")).forEach(mailspoolNumber -> {
			String index = "mailspool_" + mailspoolNumber;
			mboxMgmt.getLiteStats().stream() //
					.filter(stat -> stat.indexName.equals(index)).findFirst() //
					.ifPresent(stat -> handleIndex(domainUids, stat));
		});
	}

	private void handleIndex(List<String> domainUids, SimpleShardStats stat) {
		if (stat.mailboxes.isEmpty()) {
			ctx.info("- {}: resetting empty index (no alias & no document)", stat.indexName);
			if (apply) {
				ESearchActivator.resetIndex(stat.indexName);
			}
		} else {
			boolean noUserAlias = stat.mailboxes.stream().noneMatch(uid -> userExists(domainUids, uid));
			if (noUserAlias) {
				ctx.info("- {}: deleting index with no active user alias", stat.indexName);
				if (apply) {
					ESearchActivator.deleteIndex(stat.indexName);
				}
			} else {
				ctx.warn("- {}: keeping index, as it contains some active user alias", stat.indexName);
			}
		}
	}

	private boolean userExists(List<String> domainUids, String uid) {
		return domainUids.stream() //
				.map(domainUid -> ctx.adminApi().instance(IDirectory.class, domainUid)) //
				.anyMatch(api -> api.findByEntryUid(uid) != null);
	}

	@Override
	public Runnable forContext(CliContext ctx) {
		this.ctx = ctx;
		this.cliUtils = new CliUtils(ctx);
		return this;
	}

	public static class Reg implements ICmdLetRegistration {

		@Override
		public Optional<String> group() {
			return Optional.of("index");
		}

		@Override
		public Class<? extends ICmdLet> commandClass() {
			return PruneMailspoolCommand.class;
		}
	}

}
