/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2022
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.cli.sds;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.text.SimpleDateFormat;
import java.util.Optional;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

import io.vertx.core.buffer.Buffer;
import io.vertx.core.json.JsonArray;
import io.vertx.core.json.JsonObject;
import net.bluemind.cli.cmd.api.CliContext;
import net.bluemind.cli.cmd.api.CliException;
import net.bluemind.cli.cmd.api.ICmdLet;
import net.bluemind.cli.cmd.api.ICmdLetRegistration;
import net.bluemind.sds.dto.GetRequest;
import net.bluemind.sds.dto.SdsResponse;
import net.bluemind.sds.store.ISdsSyncStore;
import net.bluemind.sds.store.loader.SdsStoreLoader;
import net.bluemind.system.api.ISystemConfiguration;
import net.bluemind.system.api.SystemConf;
import picocli.CommandLine.Command;
import picocli.CommandLine.Option;
import picocli.CommandLine.Parameters;

@Command(name = "zip-mail-export", description = "Create a zip archive of emails belonging to a user")
public class SdsZipMailExportCommand implements ICmdLet, Runnable {

	public static class Reg implements ICmdLetRegistration {

		@Override
		public Optional<String> group() {
			return Optional.of("sds");
		}

		@Override
		public Class<? extends ICmdLet> commandClass() {
			return SdsZipMailExportCommand.class;
		}

	}

	private CliContext ctx;

	@Option(names = "--workers", description = "Parallel downloads")
	int workers = 16;

	@Parameters(paramLabel = "FILE", description = "JSON file to export")
	File jsonFile;

	private SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");

	private static final Fetched END_MARKER = new Fetched();

	@Override
	public void run() {
		JsonObject sdsMapping;
		ctx.info("Parsing {}", jsonFile);
		try {
			byte[] content = Files.readAllBytes(jsonFile.toPath());
			sdsMapping = new JsonObject(Buffer.buffer(content));
		} catch (IOException e) {
			throw new CliException(e);
		}

		String userLogin = sdsMapping.getString("login");

		ISystemConfiguration configurationApi = ctx.adminApi().instance(ISystemConfiguration.class);
		SystemConf sysConf = configurationApi.getValues();

		SdsStoreLoader loader = new SdsStoreLoader();
		ISdsSyncStore sdsStore = loader.forSysconf(sysConf).orElseThrow(() -> new CliException("Cannot load s3 store"));

		JsonArray folders = sdsMapping.getJsonArray("folders");
		int len = folders.size();
		ArrayBlockingQueue<Fetched> blocking = new ArrayBlockingQueue<>(workers);

		AtomicInteger availDownloads = new AtomicInteger();

		Thread zipThread = new Thread(() -> {
			try (ZipOutputStream zip = new ZipOutputStream(Files.newOutputStream(Paths.get(userLogin + ".zip")))) {

				do {
					Fetched fromS3;
					try {
						fromS3 = blocking.poll(30, TimeUnit.SECONDS);
						if (fromS3 == END_MARKER) {
							ctx.info("Got end marker");
							break;
						}
						if (fromS3 != null) {
							String entryName = fromS3.dir + "/" + fromS3.file.getName();
							ZipEntry ze = new ZipEntry(entryName);
							zip.putNextEntry(ze);
							Files.copy(fromS3.file.toPath(), zip);
							zip.closeEntry();
							fromS3.file.delete();
							availDownloads.incrementAndGet();
						}
					} catch (InterruptedException e) {
						break;
					}
				} while (true);
			} catch (Exception e) {
				e.printStackTrace();
				ctx.error(e.getMessage(), e);
			}
		});
		zipThread.start();

		ExecutorService downloads = Executors.newFixedThreadPool(workers);

		AtomicInteger emlNumber = new AtomicInteger();
		for (int i = 0; i < len; i++) {
			JsonObject folder = folders.getJsonObject(i);
			String fullName = folder.getString("fullName");
			JsonArray msgs = folder.getJsonArray("messages");
			int msgCount = msgs.size();
			ctx.info("On folder {} with {} message(s)", fullName, msgCount);
			for (int j = 0; j < msgCount; j++) {
				JsonObject guidAndDate = msgs.getJsonObject(j);
				String sdsKey = guidAndDate.getString("g");
				while (blocking.size() > workers / 2) {
					try {
						Thread.sleep(1);
					} catch (InterruptedException e) {
						System.exit(0);
					}
				}
				downloads.execute(() -> {
					int cur = emlNumber.incrementAndGet();
					if (cur % 10000 == 0) {
						ctx.info("Processing EML {}", cur);
					}
					File f = new File(cur + ".eml");
					GetRequest req = GetRequest.of(null, sdsKey, f.getAbsolutePath());
					SdsResponse resp = sdsStore.download(req);
					if (resp.succeeded()) {
						Fetched fetched = new Fetched();
						fetched.file = f;
						fetched.dir = fullName;
						boolean offered = false;
						while (!offered) {
							try {
								offered = blocking.offer(fetched, 1, TimeUnit.SECONDS);
							} catch (InterruptedException e) {
								return;
							}
						}
					} else {
						f.delete();
					}

				});
			}
			try {
				Thread.sleep(10);
			} catch (InterruptedException e) {
				Thread.currentThread().interrupt();
			}
		}

		ctx.info("Waiting for end of downloads...");
		downloads.shutdown();
		try {
			while (!downloads.awaitTermination(10, TimeUnit.SECONDS)) {
				// loop
			}
		} catch (InterruptedException e) {
			ctx.error("Interrupted");
			System.exit(0);
		}
		ctx.info("Download tasks completed");
		while (!blocking.offer(END_MARKER)) {
			// loop
		}
		try {
			zipThread.join();
		} catch (InterruptedException e) {
			Thread.currentThread().interrupt();
		}
		ctx.info("Zip completed with {} message(s)", availDownloads.get());
	}

	private static class Fetched {
		File file;
		String dir;
	}

	@Override
	public Runnable forContext(CliContext ctx) {
		this.ctx = ctx;
		return this;
	}
}
