/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2020
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.cti.xivo.system;

import java.util.Base64;

import net.bluemind.system.service.RegisteredExternalSystem;

public class XivoSystem extends RegisteredExternalSystem {

	public XivoSystem() {
		super("Xivo", "Xivo, An Open Source project to build your own IP telecom platform",
				AuthKind.SIMPLE_CREDENTIALS);
	}

	@Override
	public byte[] getLogo() {
		String img = "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";
		return Base64.getDecoder().decode(img);
	}

	@Override
	public boolean handles(String userAccountIdentifier) {
		return userAccountIdentifier.startsWith(super.identifier);
	}

}
