/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2022
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.cli.mail;

import java.util.Optional;

import net.bluemind.backend.mail.api.IMailConversationInternalSync;
import net.bluemind.backend.mail.replica.api.IMailReplicaUids;
import net.bluemind.cli.cmd.api.CliContext;
import net.bluemind.cli.cmd.api.ICmdLet;
import net.bluemind.cli.cmd.api.ICmdLetRegistration;
import net.bluemind.cli.utils.CliUtils;
import net.bluemind.cli.utils.CliUtils.ResolvedMailbox;
import net.bluemind.cli.utils.Tasks;
import net.bluemind.core.task.api.TaskRef;
import picocli.CommandLine.Command;
import picocli.CommandLine.Option;
import picocli.CommandLine.Parameters;

@Command(name = "conversation_sync", description = "Sync conversations internally between t_mailbox_record and t_conversation")
public class ConversationInternalDbSyncCommand implements ICmdLet, Runnable {

	@Parameters(paramLabel = "<email_address>", description = "email address (might be an alias) of the mailbox")
	public String target;

	@Option(names = "--dry", description = "Do not run the sync for real, prints out the list of missing messages in t_conversation")
	public boolean dry = false;

	private CliContext ctx;

	@Override
	public void run() {
		CliUtils cliUtils = new CliUtils(ctx);

		ResolvedMailbox mailboxByEmail = cliUtils.getMailboxByEmail(target);

		IMailConversationInternalSync syncService = ctx.adminApi().instance(IMailConversationInternalSync.class,
				IMailReplicaUids.conversationSubtreeUid(mailboxByEmail.domainUid, mailboxByEmail.mailbox.uid));

		TaskRef sync = syncService.sync(dry);
		Tasks.follow(ctx, sync, "", String.format("Failed to resync conversations of %s", target));

	}

	public static class Reg implements ICmdLetRegistration {

		@Override
		public Optional<String> group() {
			return Optional.of("mail");
		}

		@Override
		public Class<? extends ICmdLet> commandClass() {
			return ConversationInternalDbSyncCommand.class;
		}
	}

	@Override
	public Runnable forContext(CliContext ctx) {
		this.ctx = ctx;
		return this;
	}

}
