/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.lmtp.filter.imip;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.james.mime4j.codec.DecodeMonitor;
import org.apache.james.mime4j.dom.field.UnstructuredField;
import org.apache.james.mime4j.field.UnstructuredFieldImpl;
import org.apache.james.mime4j.stream.Field;
import org.apache.james.mime4j.stream.RawField;
import org.columba.ristretto.message.Address;
import org.slf4j.LoggerFactory;

import net.bluemind.calendar.api.VEventOccurrence;
import net.bluemind.calendar.helper.mail.MeetingUpdateDiff;
import net.bluemind.icalendar.api.ICalendarElement;
import net.bluemind.icalendar.api.ICalendarElement.Attendee;

public class IMIPResponse {

	public List<Field> headerFields = new ArrayList<>();

	public static IMIPResponse createNeedResponse(String itemUid, ICalendarElement calElement) {
		return createNeedResponse(itemUid, calElement, true, Collections.emptyMap());
	}

	public static IMIPResponse createCanceledResponse(String itemUid) {
		IMIPResponse ret = new IMIPResponse();

		StringBuilder eventIcsUid = new StringBuilder(itemUid);

		return getCancelHeader(ret, eventIcsUid);

	}

	public static IMIPResponse createCanceledExceptionResponse(String uid, String iso8601) {
		IMIPResponse ret = new IMIPResponse();

		StringBuilder eventIcsUid = new StringBuilder(uid);
		eventIcsUid.append("; recurid=\"" + iso8601 + "\"");

		return getCancelHeader(ret, eventIcsUid);
	}

	private static IMIPResponse getCancelHeader(IMIPResponse ret, StringBuilder value) {
		RawField rf = new RawField("X-BM-Event-Canceled", value.toString());
		UnstructuredField bmExtId = UnstructuredFieldImpl.PARSER.parse(rf, DecodeMonitor.SILENT);
		ret.headerFields = Arrays.asList(bmExtId);
		return ret;
	}

	public static IMIPResponse createRepliedResponse(String itemUid) {
		IMIPResponse ret = new IMIPResponse();

		StringBuilder eventIcsUid = new StringBuilder(itemUid);

		return getReplyHeader(ret, eventIcsUid);

	}

	public static IMIPResponse createRepliedToExceptionResponse(String itemUid, String iso8601) {
		IMIPResponse ret = new IMIPResponse();

		StringBuilder eventIcsUid = new StringBuilder(itemUid);
		eventIcsUid.append("; recurid=\"" + iso8601 + "\"");

		return getReplyHeader(ret, eventIcsUid);
	}

	private static IMIPResponse getReplyHeader(IMIPResponse ret, StringBuilder value) {
		RawField rf = new RawField("X-BM-Event-Replied", value.toString());
		UnstructuredField bmExtId = UnstructuredFieldImpl.PARSER.parse(rf, DecodeMonitor.SILENT);
		ret.headerFields = Arrays.asList(bmExtId);
		return ret;
	}
	


	public static IMIPResponse createEventResponse(String itemUid, ICalendarElement calElement, boolean needResponse,
			String calendarUid, MeetingUpdateDiff diff) {
		Map<String, String> att = (diff == null) ? Collections.emptyMap() : diff.toMap();
		return createNeedResponse(itemUid, calElement, needResponse, att);
	}

	public static IMIPResponse createNeedResponse(String itemUid, ICalendarElement calElement, boolean needResponse, Map<String, String> additionalAttributes) {
		LoggerFactory.getLogger(IMIPResponse.class).info("need resp {} {}", itemUid, needResponse);
		IMIPResponse ret = new IMIPResponse();

		StringBuilder eventIcsUid = new StringBuilder(itemUid);
		if (calElement instanceof VEventOccurrence) {
			eventIcsUid.append("; recurid=\"" + ((VEventOccurrence) calElement).recurid.iso8601 + "\"");
		}
		eventIcsUid.append("; rsvp=\"" + needResponse + "\"");
		additionalAttributes.forEach((key, value) -> eventIcsUid.append("; " + key + "=\"" + value + "\""));

		RawField rf = new RawField("X-BM-Event", eventIcsUid.toString());
		UnstructuredField bmExtId = UnstructuredFieldImpl.PARSER.parse(rf, DecodeMonitor.SILENT);
		ret.headerFields = Arrays.asList(bmExtId);
		return ret;
	}

	public static IMIPResponse createCounterResponse(String itemUid, String email, VEventOccurrence counterEvent,
			List<Attendee> proposedAttendees) {
		IMIPResponse ret = new IMIPResponse();

		StringBuilder eventIcsUid = new StringBuilder(itemUid);
		eventIcsUid.append("; originator=\"" + email + "\"");

		if (counterEvent.recurid != null) {
			eventIcsUid.append("; recurid=\"" + counterEvent.recurid.iso8601 + "\"");
		}

		RawField rf = new RawField("X-BM-Event-Countered", eventIcsUid.toString());
		UnstructuredField bmExtId = UnstructuredFieldImpl.PARSER.parse(rf, DecodeMonitor.SILENT);
		ret.headerFields = Arrays.asList(bmExtId);

		if (!proposedAttendees.isEmpty()) {
			ret.headerFields = new ArrayList<>(ret.headerFields);
			String attendeeList = String.join(",", proposedAttendees.stream()
					.map(att -> new Address(att.commonName, att.mailto).toString()).collect(Collectors.toList()));
			RawField attendeeHeader = new RawField("X-BM-Counter-Attendee", attendeeList);
			UnstructuredField attendeeHeaderField = UnstructuredFieldImpl.PARSER.parse(attendeeHeader,
					DecodeMonitor.SILENT);
			ret.headerFields.add(attendeeHeaderField);
		}

		return ret;

	}

	public static IMIPResponse createEmptyResponse() {
		return new IMIPResponse();
	}

}
